#include "mdbsave.h"

#include "common.h"

#include <mail/notsolitesrv/src/util/optional.h>

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/transform.hpp>

#include <algorithm>
#include <cstdint>
#include <iterator>
#include <utility>

namespace NNotSoLiteSrv::NMetaSaveOp {

using NUtil::ConvertOptional;

bool RecipientSucceeded(const TDeliveryId& deliveryId,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    if (!rulesApplierResponse) {
        return true;
    }

    const auto failedRecipient = std::find(rulesApplierResponse->FailedRecipients.cbegin(),
        rulesApplierResponse->FailedRecipients.cend(), deliveryId);
    return failedRecipient == rulesApplierResponse->FailedRecipients.cend();
}

bool RecipientToSaveForAvailable(const TRecipientMap& recipients,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    auto predicate =
        [&](const auto& element){return RecipientSucceeded(element.first, rulesApplierResponse);};
    return std::any_of(recipients.cbegin(), recipients.cend(), std::move(predicate));
}

NMdbSave::TUser MakeMdbSaveUser(const TUser& user) {
    NMdbSave::TUser mdbSaveUser;
    mdbSaveUser.Uid = std::to_string(user.uid);
    mdbSaveUser.Suid = ConvertOptional(user.suid);
    return mdbSaveUser;
}

NMdbSave::TRequestLabel MakeMdbSaveRequestLabel(const TLabel& label) {
    return {label.name, label.type};
}

std::optional<NMdbSave::TRequestLabels> MakeMdbSaveRequestLabels(const std::vector<TLabel>& labels) {
    if (labels.empty()) {
        return {};
    }

    NMdbSave::TRequestLabels mdbSaveRequestLabels;
    auto transformer = [](const auto& label){return MakeMdbSaveRequestLabel(label);};
    std::transform(labels.cbegin(), labels.cend(), std::back_inserter(mdbSaveRequestLabels),
        std::move(transformer));
    return mdbSaveRequestLabels;
}

NMdbSave::TStorage MakeMdbSaveStorage(const TDeliveryId& deliveryId, const TRequest& request) {
    NMdbSave::TStorage storage;
    const auto& recipient = request.recipients.at(deliveryId);
    if (recipient.params.stid) {
        storage.Stid = *recipient.params.stid;
        storage.Offset = ConvertOptional(recipient.params.offset_diff);
    } else {
        storage.Stid = request.stid;
    }

    return storage;
}

NMdbSave::THeaders MakeMdbSaveHeaders(const TDeliveryId& deliveryId, const TRequest& request) {
    NMdbSave::THeaders headers;
    headers.RecievedDate = request.envelope.received_date;
    headers.Date = request.message.date;
    headers.Subject = request.message.subject;
    headers.MsgId = request.message.message_id;
    headers.ReplyTo = ConvertOptional(request.message.reply_to);
    headers.InReplyTo = ConvertOptional(request.message.in_reply_to);
    headers.From = request.message.from;
    headers.To = request.message.to;
    if (request.message.cc && (!request.message.cc->empty())) {
        headers.Cc = *request.message.cc;
    }

    const auto& recipient = request.recipients.at(deliveryId);
    if (recipient.params.bcc && (!recipient.params.bcc->empty())) {
        headers.Bcc = *recipient.params.bcc;
    }

    return headers;
}

NMdbSave::TAttachment MakeMdbSaveAttachment(const std::string& hid, const TAttachment& attachment) {
    return {hid, attachment.name, attachment.type, static_cast<size_t>(attachment.size)};
}

std::optional<std::vector<NMdbSave::TAttachment>> MakeMdbSaveAttachments(
    const NMetaSaveOp::TMessage& message)
{
    if (message.attachments && (!message.attachments->empty())) {
        std::vector<NMdbSave::TAttachment> attachments;
        auto transformer =
            [](const auto& element){return MakeMdbSaveAttachment(element.first, element.second);};
        std::transform(message.attachments->cbegin(), message.attachments->cend(), std::back_inserter(
            attachments), std::move(transformer));
        return attachments;
    } else {
        return {};
    }
}

NMdbSave::TMimePart MakeMdbSaveMimePart(const std::string& hid, const NMetaSaveOp::TPart& part) {
    return {
        hid,
        part.content_type,
        part.content_subtype,
        ConvertOptional(part.boundary),
        ConvertOptional(part.name),
        part.charset,
        part.encoding,
        ConvertOptional(part.content_disposition),
        ConvertOptional(part.file_name),
        ConvertOptional(part.content_id),
        static_cast<size_t>(part.offset),
        static_cast<size_t>(part.length)
    };
}

std::optional<std::vector<NMdbSave::TMimePart>> MakeMdbSaveMimeParts(const TPartMap& parts) {
    if (parts.empty()) {
        return {};
    }

    std::vector<NMdbSave::TMimePart> mdbSaveMimeParts;
    auto transformer =
        [](const auto& element){return MakeMdbSaveMimePart(element.first, element.second);};
    std::transform(parts.cbegin(), parts.cend(), std::back_inserter(mdbSaveMimeParts),
        std::move(transformer));
    return mdbSaveMimeParts;
}

NMdbSave::TThreadInfo MakeMdbSaveThreadInfo(const NMthr::TThreadInfo& threadInfo) {
    return {
        {threadInfo.Hash.Namespace, threadInfo.Hash.Value},
        {threadInfo.Limits.Days, threadInfo.Limits.Count},
        threadInfo.Rule,
        threadInfo.ReferenceHashes,
        threadInfo.MessageIds,
        threadInfo.InReplyToHash,
        threadInfo.MessageIdHash
    };
}

NMdbSave::TMessage MakeMdbSaveMessage(const TDeliveryId& deliveryId, const TRequest& request,
    const NFirstline::TFirstlineResult& firstline, const NMthr::TMthrResponse& mthr)
{
    NMdbSave::TMessage mdbSaveMessage;
    const auto& recipient = request.recipients.at(deliveryId);
    mdbSaveMessage.OldMid = ConvertOptional(recipient.params.old_mid);

    if (recipient.params.external_imap_id) {
        mdbSaveMessage.ExtImapId = static_cast<int64_t>(*recipient.params.external_imap_id);
    }

    if (firstline) {
        mdbSaveMessage.Firstline = firstline->Firstline;
    }

    mdbSaveMessage.Size = static_cast<size_t>(request.message.size);
    mdbSaveMessage.Lids = recipient.params.lids;
    mdbSaveMessage.LabelSymbols = recipient.params.label_symbols;
    mdbSaveMessage.Labels = MakeMdbSaveRequestLabels(recipient.params.labels);
    mdbSaveMessage.Tab = ConvertOptional(recipient.params.tab);

    mdbSaveMessage.Storage = MakeMdbSaveStorage(deliveryId, request);
    mdbSaveMessage.Headers = MakeMdbSaveHeaders(deliveryId, request);
    mdbSaveMessage.Attachments = MakeMdbSaveAttachments(request.message);
    mdbSaveMessage.MimeParts = MakeMdbSaveMimeParts(request.message.parts);
    mdbSaveMessage.ThreadInfo = MakeMdbSaveThreadInfo(mthr.ThreadInfo);
    return mdbSaveMessage;
}

std::optional<NMdbSave::TRequestFolder> MakeMdbSaveRequestDestinationFolder(const TDeliveryId& deliveryId,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    if (rulesApplierResponse) {
        const auto element = rulesApplierResponse->AppliedRules.find(deliveryId);
        if (element != rulesApplierResponse->AppliedRules.end()) {
            const auto& destFolder = element->second.DestFolder;
            if (!destFolder.Fid.empty() || !destFolder.Path.empty()) {
                NMdbSave::TRequestFolder mdbSaveDestFolder;
                mdbSaveDestFolder.Fid = MakeOptionalRange(destFolder.Fid);

                if (!destFolder.Path.empty()) {
                    mdbSaveDestFolder.Path = {destFolder.Path, {}};
                }

                return mdbSaveDestFolder;
            }
        }
    }

    return {};
}

std::optional<NMdbSave::TRequestFolder> MakeMdbSaveRequestOriginalFolder(const TParams& params) {
    if (params.folder) {
        const auto& originalFolder = *params.folder;
        if ((originalFolder.fid && !originalFolder.fid->empty()) ||
            (originalFolder.path && !originalFolder.path->path.empty()))
        {
            NMdbSave::TRequestFolder mdbSaveOriginalFolder;
            mdbSaveOriginalFolder.Fid = ConvertOptional(originalFolder.fid);

            if (originalFolder.path && !originalFolder.path->path.empty()) {
                NMdbSave::TPath mdbSaveOriginalFolderPath;
                const auto& originalFolderPath = *originalFolder.path;
                mdbSaveOriginalFolderPath.Path = originalFolderPath.path;
                mdbSaveOriginalFolderPath.Delimeter = MakeOptionalRange(originalFolderPath.delim);
                mdbSaveOriginalFolder.Path = std::move(mdbSaveOriginalFolderPath);
            }

            return mdbSaveOriginalFolder;
        }
    }

    return {};
}

NMdbSave::TRequestFolders MakeMdbSaveRequestFolders(const TDeliveryId& deliveryId, const TRequest& request,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    NMdbSave::TRequestFolders folders;
    folders.Destination = MakeMdbSaveRequestDestinationFolder(deliveryId, rulesApplierResponse);
    folders.Original = MakeMdbSaveRequestOriginalFolder(request.recipients.at(deliveryId).params);
    return folders;
}

NMdbSave::TDuplicates MakeMdbSaveActionDuplicates(const TParams& params) {
    return {params.ignore_duplicates, params.remove_duplicates};
}

NMdbSave::TFolderActions MakeMdbSaveOriginalFolderActions(const TParams& params) {
    return {params.store_as_deleted, std::to_string(params.no_such_folder_action)};
}

std::optional<NMdbSave::TFolderActions> MakeMdbSaveRulesApplierFolderActions(const TDeliveryId& deliveryId,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    if (rulesApplierResponse) {
        const auto element = rulesApplierResponse->AppliedRules.find(deliveryId);
        if (element != rulesApplierResponse->AppliedRules.end()) {
            NMdbSave::TFolderActions folderActions;
            const auto& appliedRules = element->second;
            folderActions.StoreAsDeleted = appliedRules.StoreAsDeleted;
            folderActions.NoSuchFolder = MakeOptionalRange(appliedRules.NoSuchFolderAction);
            return folderActions;
        }
    }

    return {};
}

NMdbSave::TActions MakeMdbSaveActions(const TDeliveryId& deliveryId, const TRequest& request,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    NMdbSave::TActions actions;
    const auto& params = request.recipients.at(deliveryId).params;
    actions.Duplicates = MakeMdbSaveActionDuplicates(params);
    actions.UseFilters = params.use_filters;
    actions.DisablePush = params.disable_push;
    actions.Original = MakeMdbSaveOriginalFolderActions(params);
    actions.RulesApplied = MakeMdbSaveRulesApplierFolderActions(deliveryId, rulesApplierResponse);
    return actions;
}

std::optional<NMdbSave::TRequestAddedLids> MakeMdbSaveAddedLids(const TDeliveryId& deliveryId,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    if (rulesApplierResponse) {
        const auto element = rulesApplierResponse->AppliedRules.find(deliveryId);
        if (element != rulesApplierResponse->AppliedRules.end()) {
            return element->second.Lids;
        }
    }

    return {};
}

std::optional<NMdbSave::TRequestAddedSymbols> MakeMdbSaveAddedSymbols(const TDeliveryId& deliveryId,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    if (rulesApplierResponse) {
        const auto element = rulesApplierResponse->AppliedRules.find(deliveryId);
        if (element != rulesApplierResponse->AppliedRules.end()) {
            return element->second.LabelSymbols;
        }
    }

    return {};
}

NMdbSave::TMdbSaveRequestRcpt MakeMdbSaveRcpt(
    const TDeliveryId& deliveryId,
    const TRequest& request,
    const NFirstline::TFirstlineResult& firstline,
    const NMthr::TMthrResponse& mthr,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    NMdbSave::TMdbSaveRequestRcpt mdbSaveRcpt;
    const auto& recipient = request.recipients.at(deliveryId);
    mdbSaveRcpt.User = MakeMdbSaveUser(recipient.user);
    mdbSaveRcpt.Message = MakeMdbSaveMessage(deliveryId, request, firstline, mthr);
    mdbSaveRcpt.Folders = MakeMdbSaveRequestFolders(deliveryId, request, rulesApplierResponse);
    mdbSaveRcpt.Actions = MakeMdbSaveActions(deliveryId, request, rulesApplierResponse);
    mdbSaveRcpt.AddedLids = MakeMdbSaveAddedLids(deliveryId, rulesApplierResponse);
    mdbSaveRcpt.AddedSymbols = MakeMdbSaveAddedSymbols(deliveryId, rulesApplierResponse);
    mdbSaveRcpt.Imap = recipient.params.imap;
    return mdbSaveRcpt;
}

NMdbSave::TMdbSaveRequestRcptNode MakeMdbSaveRcptNode(
    const TDeliveryId& deliveryId,
    const TRequest& request,
    const NFirstline::TFirstlineResult& firstline,
    const NMthr::TMthrResponse& mthr,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    NMdbSave::TMdbSaveRequestRcptNode mdbSaveRcptNode;
    mdbSaveRcptNode.Id = deliveryId;
    mdbSaveRcptNode.Rcpt = MakeMdbSaveRcpt(deliveryId, request, firstline, mthr, rulesApplierResponse);
    return mdbSaveRcptNode;
}

std::vector<NMdbSave::TMdbSaveRequestRcptNode> MakeMdbSaveRcpts(
    const TRequest& request,
    const NFirstline::TFirstlineResult& firstline,
    const NMthr::TMthrResponse& mthr,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    auto filter = [&](const auto& element){return RecipientSucceeded(element.first, rulesApplierResponse);};
    std::vector<NMdbSave::TMdbSaveRequestRcptNode> rcpts;
    auto transformer = [&](const auto& element) {
        return MakeMdbSaveRcptNode(element.first, request, firstline, mthr, rulesApplierResponse);
    };

    boost::transform(request.recipients | boost::adaptors::filtered(std::move(filter)),
        std::back_inserter(rcpts), std::move(transformer));
    return rcpts;
}

NMdbSave::TMdbSaveRequest MakeMdbSaveRequest(
    const TRequest& request,
    const NFirstline::TFirstlineResult& firstline,
    const NMthr::TMthrResponse& mthr,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse)
{
    NMdbSave::TMdbSaveRequest mdbSaveRequest;
    mdbSaveRequest.Rcpts = MakeMdbSaveRcpts(request, firstline, mthr, rulesApplierResponse);
    mdbSaveRequest.Sync = request.sync_dlv;
    return mdbSaveRequest;
}

std::optional<std::reference_wrapper<const NMdbSave::TMdbSaveResponseRcpt>>
    FindMdbSaveResponseRcptByDeliveryId(const TDeliveryId& deliveryId,
        const std::vector<NMdbSave::TMdbSaveResponseRcptNode>& rcpts)
{
    auto predicate = [&](const auto& rcptNode){return rcptNode.Id == deliveryId;};
    const auto rcptNode = std::find_if(rcpts.cbegin(), rcpts.cend(), std::move(predicate));
    if (rcptNode == rcpts.end()) {
        return {};
    } else {
        return std::cref(rcptNode->Rcpt);
    }
}

}
