#include "response.h"

#include "common.h"
#include "mdbsave.h"

#include <mail/notsolitesrv/src/util/optional.h>

#include <boost/lexical_cast.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/transform.hpp>

#include <algorithm>
#include <iterator>

namespace NNotSoLiteSrv::NMetaSaveOp {

using NUtil::ConvertOptional;

std::map<TDeliveryId, std::string> MakeFailureStatusByDeliveryIdMap(
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse,
    const NMdbSave::TMdbSaveResult& mdbSave)
{
    std::map<TDeliveryId, std::string> failureStatusByDeliveryId;
    if (rulesApplierResponse) {
        auto transformer =
            [](const auto& failedRecipient){return std::make_pair(failedRecipient, "temp error");};
        std::transform(
            rulesApplierResponse->FailedRecipients.cbegin(),
            rulesApplierResponse->FailedRecipients.cend(),
            std::inserter(failureStatusByDeliveryId, failureStatusByDeliveryId.end()),
            std::move(transformer));
    }

    if (mdbSave) {
        auto filter = [&](const auto& rcptNode){return rcptNode.Rcpt.Status != "ok";};
        auto transformer =
            [](const auto& rcptNode){return std::make_pair(rcptNode.Id, rcptNode.Rcpt.Status);};
        boost::transform(
            mdbSave->Rcpts | boost::adaptors::filtered(std::move(filter)),
            std::inserter(failureStatusByDeliveryId, failureStatusByDeliveryId.end()),
            std::move(transformer));
    }

    return failureStatusByDeliveryId;
}

EStatus MakeRecipientResponseStatus(const std::string& status) {
    if (status == "ok") {
        return EStatus::Ok;
    } else if (status == "temp error") {
        return EStatus::TemporaryError;
    } else if (status == "perm error") {
        return EStatus::PermanentError;
    } else {
        return EStatus::Unknown;
    }
}

boost::optional<std::vector<TAutoReply>> MakeRecipientResponseReply(
    const std::vector<NRulesApplier::TAutoreply>& replies)
{
    if (replies.empty()) {
        return {};
    }

    std::vector<TAutoReply> responseReplies;
    auto transformer = [](const auto& reply){return TAutoReply{reply.Address, reply.Body};};
    std::transform(replies.cbegin(), replies.cend(), std::back_inserter(responseReplies),
        std::move(transformer));
    return responseReplies;
}

boost::optional<std::vector<TForward>> MakeRecipientResponseForward(
    const std::vector<std::string>& forwards)
{
    if (forwards.empty()) {
        return {};
    }

    std::vector<TForward> responseForwards;
    auto transformer = [](const auto& forward){return TForward{forward};};
    std::transform(forwards.cbegin(), forwards.cend(), std::back_inserter(responseForwards),
        std::move(transformer));
    return responseForwards;
}

boost::optional<std::vector<TNotify>> MakeRecipientResponseNotify(
    const std::vector<std::string>& notifies)
{
    if (notifies.empty()) {
        return {};
    }

    std::vector<TNotify> responseNotifies;
    auto transformer = [](const auto& notify){return TNotify{notify};};
    std::transform(notifies.cbegin(), notifies.cend(), std::back_inserter(responseNotifies),
        std::move(transformer));
    return responseNotifies;
}

uint32_t MakeMdbCommitResponseImapId(const std::optional<std::string>& imapId) {
    if (imapId) {
        std::uint32_t convertedImapId{0};
        if (boost::conversion::try_lexical_convert(*imapId, convertedImapId)) {
            return convertedImapId;
        }
    }

    return 0;
}

boost::optional<NMetaSaveOp::TResolvedFolder> MakeResolvedFolder(
    const std::optional<NMdbSave::TResolvedFolder>& folder)
{
    if (!folder) {
        return {};
    }

    NMetaSaveOp::TResolvedFolder resolvedFolder;
    resolvedFolder.fid = folder->Fid;
    resolvedFolder.name = ConvertOptional(MakeOptionalRange(folder->Name));
    resolvedFolder.type = ConvertOptional(MakeOptionalRange(folder->Type));
    resolvedFolder.type_code = folder->TypeCode;
    return resolvedFolder;
}

TResolvedLabel MakeResolvedLabel(const NMdbSave::TResponseLabel& label) {
    return {label.Lid, ConvertOptional(MakeOptionalRange(label.Symbol))};
}

boost::optional<std::vector<TResolvedLabel>> MakeResolvedLabels(
    const std::optional<std::vector<NMdbSave::TResponseLabel>>& labels)
{
    if (!labels) {
        return {};
    }

    std::vector<TResolvedLabel> mdbCommitResponseLabels;
    auto transformer = [](const auto& label){return MakeResolvedLabel(label);};
    std::transform(labels->cbegin(), labels->cend(), std::back_inserter(mdbCommitResponseLabels),
        std::move(transformer));
    return mdbCommitResponseLabels;
}

boost::optional<TMdbCommitResponse> MakeMdbCommitResponse(const TDeliveryId& deliveryId,
    const NMdbSave::TMdbSaveResult& mdbSave)
{
    if (mdbSave) {
        const auto mdbSaveResponseRcpt = FindMdbSaveResponseRcptByDeliveryId(deliveryId, mdbSave->Rcpts);
        if (mdbSaveResponseRcpt) {
            TMdbCommitResponse response;
            const auto& rcpt = mdbSaveResponseRcpt->get();
            response.uid = rcpt.Uid;
            response.status = MakeRecipientResponseStatus(rcpt.Status);
            response.description = ConvertOptional(rcpt.Description);
            response.mid = ConvertOptional(rcpt.Mid);
            response.imap_id = MakeMdbCommitResponseImapId(rcpt.ImapId);
            response.duplicate = rcpt.Duplicate.value_or(false);
            response.tid = ConvertOptional(rcpt.Tid);
            response.folder = MakeResolvedFolder(rcpt.Folder);
            response.labels = MakeResolvedLabels(rcpt.Labels);
            return response;
        }
    }

    return {};
}

TRecipientResponse MakeRecipientResponse(
    const std::map<TDeliveryId, std::string>& failureStatusByDeliveryId,
    const TDeliveryId& deliveryId,
    bool useFilters,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse,
    const NMdbSave::TMdbSaveResult& mdbSave)
{
    TRecipientResponse response;
    if (failureStatusByDeliveryId.find(deliveryId) != failureStatusByDeliveryId.end()) {
        response.status = MakeRecipientResponseStatus(failureStatusByDeliveryId.at(deliveryId));
    } else {
        response.status = EStatus::Ok;

        if (useFilters && rulesApplierResponse) {
            const auto& appliedRules = rulesApplierResponse->AppliedRules.at(deliveryId);
            response.reply = MakeRecipientResponseReply(appliedRules.Replies);
            response.forward = MakeRecipientResponseForward(appliedRules.Forwards);
            response.notify = MakeRecipientResponseNotify(appliedRules.Notifies);
            response.rule_ids = ConvertOptional(MakeOptionalRange(appliedRules.RuleIds));
        }
    }

    response.mdb_commit = MakeMdbCommitResponse(deliveryId, mdbSave);
    return response;
}

TResponse::TMdbResponses MakeMdbResponses(
    const TRecipientMap& recipients,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse,
    const NMdbSave::TMdbSaveResult& mdbSave)
{
    TResponse::TMdbResponses mdbResponses;
    const auto failureStatusByDeliveryId = MakeFailureStatusByDeliveryIdMap(rulesApplierResponse, mdbSave);
    auto transformer = [&](const auto& element) {
        auto recipientResponse = MakeRecipientResponse(failureStatusByDeliveryId, element.first,
            element.second.params.use_filters, rulesApplierResponse, mdbSave);
        return std::make_pair(element.first, std::move(recipientResponse));
    };

    std::transform(recipients.cbegin(), recipients.cend(), std::inserter(mdbResponses, mdbResponses.end()),
        std::move(transformer));
    return mdbResponses;
}

TResponse MakeResponse(
    const TRecipientMap& recipients,
    const std::optional<NRulesApplier::TRulesApplierResponse>& rulesApplierResponse,
    const NMdbSave::TMdbSaveResult& mdbSave)
{
    TResponse response;
    response.mdb_responses = MakeMdbResponses(recipients, rulesApplierResponse, mdbSave);
    return response;
}

}
