#include "tupita.h"

#include "common.h"
#include "furita.h"

#include <boost/range/adaptor/filtered.hpp>

#include <algorithm>
#include <iterator>
#include <utility>

namespace NNotSoLiteSrv::NMetaSaveOp {

NTupita::TLabelsInfoMap MakeLabelsInfoMap(const std::string& domainLabel) {
    NTupita::TTupitaMessageLabel messageLabel;
    messageLabel.Name = domainLabel;
    messageLabel.IsSystem = false;
    messageLabel.IsUser = false;
    messageLabel.Type.Title = "social";
    NTupita::TLabelsInfoMap labelsInfo;
    labelsInfo["FAKE_DOMAIN_LABEL"] = std::move(messageLabel);
    return labelsInfo;
}

NTupita::TTupitaMessage MakeTupitaMessage(const TRequest& request,
    const NFirstline::TFirstlineResult& firstline)
{
    NTupita::TTupitaMessage message;
    message.Subject = request.message.subject;
    message.From = request.message.from;
    message.To = request.message.to;
    if (request.message.cc) {
        message.Cc = *request.message.cc;
    }

    message.Stid = request.stid;
    message.Spam = request.message.spam;
    message.Types = request.types;
    if (request.message.attachments) {
        message.AttachmentsCount = request.message.attachments->size();
    }

    if (request.domain_label) {
        message.LabelsInfo = MakeLabelsInfoMap(*request.domain_label);
    }

    if (firstline) {
        message.Firstline = MakeOptionalRange(firstline->Firstline);
    }

    return message;
}

NTupita::TTupitaQuery MakeTupitaQuery(const NFurita::TFuritaRule& rule) {
    return {rule.Id, rule.Query, rule.Stop};
}

std::vector<NTupita::TTupitaUser> MakeTupitaUsers(const TRequest& request, TUid uid,
    const std::vector<NFurita::TFuritaRule>& rules)
{
    NTupita::TTupitaUser tupitaUser;
    tupitaUser.Uid = uid;

    const auto furitaRuleByPriorityAndId{MakeFuritaRuleByPriorityAndIdMap(rules)};
    auto transformer{[](const auto& element){return MakeTupitaQuery(element.second);}};
    std::transform(furitaRuleByPriorityAndId.cbegin(), furitaRuleByPriorityAndId.cend(),
        std::back_inserter(tupitaUser.Queries), std::move(transformer));

    const auto recipient{FindRecipientByUid(request.recipients, uid)};
    tupitaUser.Spam = recipient ? recipient->get().params.spam : request.message.spam;
    return {std::move(tupitaUser)};
}

NTupita::TTupitaCheckRequest MakeTupitaCheckRequest(const TRequest& request,
    const NFirstline::TFirstlineResult& firstline, TUid uid, const std::vector<NFurita::TFuritaRule>& rules)
{
    NTupita::TTupitaCheckRequest tupitaCheckRequest;
    tupitaCheckRequest.Message = MakeTupitaMessage(request, firstline);
    tupitaCheckRequest.Users = MakeTupitaUsers(request, uid, rules);
    return tupitaCheckRequest;
}

}
