#include "client_impl.h"

#include <mail/notsolitesrv/src/errors.h>
#include <mail/notsolitesrv/src/http/util.h>
#include <mail/notsolitesrv/src/tskv/log.h>
#include <mail/notsolitesrv/src/config/msearch.h>
#include <mail/notsolitesrv/src/msearch/types/reflection/response.h>
#include <yamail/data/deserialization/yajl.h>
#include <yplatform/encoding/url_encode.h>

#include <boost/asio/post.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <utility>
#include <numeric>
#include <string>
#include <string_view>
#include <algorithm>
#include <functional>

namespace NNotSoLiteSrv::NMSearch {

static const std::string MSEARCH { "MSEARCH" };

TMSearchClientImpl::TMSearchClientImpl(TContextPtr ctx, TClusterCallPtr clusterCall, TTvmModulePtr tvmModule)
    : Ctx(std::move(ctx))
    , ClusterCall(std::move(clusterCall))
    , TvmModule(tvmModule) {
}

void TMSearchClientImpl::SubscriptionStatus(boost::asio::io_context& ioContext, const TSubscriptionStatusRequest& request,
        TSubscriptionStatusCallback callback) const {
    try {
        auto httpRequest = MakeSubscriptionStatusHttpRequest(request);
        ClusterCall->async_run(
            Ctx->GetTaskContext(MSEARCH),
            std::move(httpRequest),
            [self = shared_from_this(), executor = ioContext.get_executor(), callback = std::move(callback)](auto ec, auto response) {
                boost::asio::post(
                    executor,
                    std::bind(&TMSearchClientImpl::ProcessSubscriptionStatusResponse, self, ec, std::move(response), std::move(callback))
                );
            }
        );
    } catch (EError error) {
        boost::asio::post(ioContext, [error, callback = std::move(callback)]() {
            callback(make_error_code(error), {});
        });
    }
}

THttpRequest TMSearchClientImpl::MakeSubscriptionStatusHttpRequest(const TSubscriptionStatusRequest& request) const {
    auto encodeUids = [](const auto& uids) {
        using boost::algorithm::join;
        using boost::adaptors::transformed;
        return yplatform::url_encode<std::string>(
            join(uids | transformed([](auto uid) { return std::to_string(uid); }), ",")
        );
    };

    auto url = std::string("/api/async/mail/subscriptions/status")
        + "?uid=" + encodeUids(request.Uids)
        + "&opt_in_subs_uid=" + encodeUids(request.OptInSubsUids)
        + "&email=" + yplatform::url_encode<std::string>(request.SubscriptionEmail);

    std::string headers;
    if (Ctx->GetConfig()->MSearch->UseTvm) {
        const auto ticket = GetTvmServiceTicket();
        headers = MakeTvmServiceTicketHeader(ticket);
    }

    return THttpRequest::GET(std::move(url), std::move(headers));
}

std::string TMSearchClientImpl::GetTvmServiceTicket() const {
    std::string ticket;
    if (auto ec = TvmModule->get_service_ticket(Ctx->GetConfig()->MSearch->Service, ticket)) {
        NSLS_LOG_CTX_ERROR(log::error_code = ec, log::where_name = MSEARCH);
        throw EError::TvmServiceTicketError;
    }

    return ticket;
}

void TMSearchClientImpl::ProcessSubscriptionStatusResponse(TErrorCode ec, THttpResponse response,
        TSubscriptionStatusCallback callback) const {
    if (ec || (response.status != 200)) {
        ProcessError(std::move(ec), std::move(response), std::move(callback));
    } else {
        auto subscriptionStatusResult = ParseSubscriptionStatusResponse(response.body);
        if (!subscriptionStatusResult) {
            ec = make_error_code(EError::MSearchResponseParseError);
        }

        callback(std::move(ec), std::move(subscriptionStatusResult));
    }
}

void TMSearchClientImpl::ProcessError(TErrorCode ec, THttpResponse response, TSubscriptionStatusCallback callback) const {
    if (ec) {
        NSLS_LOG_CTX_ERROR(log::error_code = ec, log::where_name = MSEARCH);
    } else {
        const bool isNonRetriable = ((response.status / 100) != 5);
        ec = make_error_code(isNonRetriable ? EError::HttpNonRetryableStatus : EError::HttpRetriesExceeded);
        NSLS_LOG_CTX_ERROR(log::message = MakeErrorStatusMessage(std::move(response.body)), log::error_code = ec,
            log::where_name = MSEARCH);
    }

    callback(std::move(ec), {});
}

std::string TMSearchClientImpl::MakeErrorStatusMessage(std::string responseBody) const {
    constexpr std::string::size_type maxLength = 1000U;
    auto v = std::string_view(responseBody.c_str(), std::min(maxLength, responseBody.size()));
    return responseBody.erase(v.substr(0, v.find_first_of("\r\n")).length());
}

TSubscriptionStatusResult TMSearchClientImpl::ParseSubscriptionStatusResponse(const std::string& responseBody) const {
    try {
        return yamail::data::deserialization::fromJson<TSubscriptionStatusResponse>(responseBody);
    } catch (const std::exception& ex) {
        NSLS_LOG_CTX_ERROR(log::message = "failed to parse subscription status response", log::exception = ex,
            log::where_name = MSEARCH);
    }

    return {};
}

} // namespace NNotSoLiteSrv::NMSearch
