#include "client_impl.h"

#include <mail/notsolitesrv/src/errors.h>
#include <mail/notsolitesrv/src/http/util.h>
#include <mail/notsolitesrv/src/tskv/log.h>
#include <mail/notsolitesrv/src/config/msettings.h>
#include <mail/notsolitesrv/src/msettings/types/reflection/response.h>
#include <yamail/data/deserialization/yajl.h>
#include <yplatform/encoding/url_encode.h>

#include <boost/asio/post.hpp>
#include <boost/algorithm/string/join.hpp>

#include <utility>
#include <string>
#include <functional>
#include <exception>

namespace NNotSoLiteSrv::NMSettings {

static const std::string MSETTINGS { "MSETTINGS" };
static const std::string HANDLE_PARAMS { "/get_params" };
static const std::string HANDLE_PROFILE { "/get_profile" };

TMSettingsClientImpl::TMSettingsClientImpl(boost::asio::io_context& io, TConfigPtr config, TClusterCallPtr clusterCall,
        TTvmModulePtr tvmModule)
    : Io(io)
    , Config(std::move(config))
    , ClusterCall(std::move(clusterCall))
    , TvmModule(std::move(tvmModule)) {
}

void TMSettingsClientImpl::GetParams(TContextPtr ctx, const TParamsRequest& request, TParamsCallback callback) const {
    Get(ctx, request, callback, HANDLE_PARAMS);
}

void TMSettingsClientImpl::GetProfile(TContextPtr ctx, const TParamsRequest& request, TParamsCallback callback) const {
    Get(ctx, request, callback, HANDLE_PROFILE);
}

void TMSettingsClientImpl::Get(TContextPtr ctx, const TParamsRequest& request,
    TParamsCallback callback, const std::string& handle) const
{
    auto [error, httpRequest] = MakeGetParamsHttpRequest(ctx, request, handle);
    if (error != EError::Ok) {
        boost::asio::post(Io, [error = error, callback = std::move(callback)]() {
            callback(make_error_code(error), {});
        });
        return;
    }

    ClusterCall->async_run(
        ctx->GetTaskContext(MSETTINGS),
        std::move(httpRequest),
        [self = shared_from_this(), ctx, executor = Io.get_executor(), callback = std::move(callback)](auto ec, auto response) {
            boost::asio::post(
                executor,
                std::bind(&TMSettingsClientImpl::ProcessGetParamsResponse, self, ctx, ec, std::move(response), std::move(callback))
            );
        }
    );
}

std::pair<EError, THttpRequest> TMSettingsClientImpl::MakeGetParamsHttpRequest(
    TContextPtr ctx, const TParamsRequest& request, const std::string& handle
) const {
    using boost::algorithm::join;
    auto url = handle
        + "?uid=" + yplatform::url_encode<std::string>(std::to_string(request.Uid))
        + "&settings_list=" + yplatform::url_encode<std::string>(join(request.Params, "\r"))
        + "&service=nsls";
    if (HANDLE_PROFILE == handle) {
        url += "&ask_validator=y";
    }

    std::string headers;
    if (Config->MSettings->UseTvm) {
        const auto& [error, ticket] = GetTvmServiceTicket(ctx);
        if (error != EError::Ok) {
            return {error, {}};
        }

        headers = MakeTvmServiceTicketHeader(ticket);
    }

    return {EError::Ok, THttpRequest::GET(std::move(url), std::move(headers))};
}

std::pair<EError, std::string> TMSettingsClientImpl::GetTvmServiceTicket(TContextPtr ctx) const {
    std::string ticket;
    if (auto ec = TvmModule->get_service_ticket(Config->MSettings->Service, ticket)) {
        NSLS_LOG_ERROR(ctx, log::error_code = ec, log::where_name = MSETTINGS);
        return {EError::TvmServiceTicketError, {}};
    }

    return {EError::Ok, ticket};
}

void TMSettingsClientImpl::ProcessGetParamsResponse(TContextPtr ctx, TErrorCode ec, THttpResponse response,
        TParamsCallback callback) const {
    if (ec || (response.status != 200)) {
        ProcessError(ctx, std::move(ec), std::move(response), std::move(callback));
    } else {
        auto paramsResult = ParseGetParamsResponse(ctx, response.body);
        if (!paramsResult) {
            ec = make_error_code(EError::MSettingsResponseParseError);
        }

        callback(std::move(ec), std::move(paramsResult));
    }
}

void TMSettingsClientImpl::ProcessError(TContextPtr ctx, TErrorCode ec, THttpResponse response, TParamsCallback callback) const {
    if (ec) {
        NSLS_LOG_ERROR(ctx, log::error_code = ec, log::where_name = MSETTINGS);
    } else {
        const bool isNonRetriable = ((response.status / 100) != 5);
        ec = make_error_code(isNonRetriable ? EError::HttpNonRetryableStatus : EError::HttpRetriesExceeded);
        NSLS_LOG_ERROR(ctx, log::message = std::move(response.body), log::error_code = ec, log::where_name = MSETTINGS);
    }

    callback(std::move(ec), {});
}

TParamsResult TMSettingsClientImpl::ParseGetParamsResponse(TContextPtr ctx, const std::string& responseBody) const {
    try {
        return yamail::data::deserialization::fromJson<TFullResponse>(responseBody).settings.single_settings;
    } catch (const std::exception& ex) {
        NSLS_LOG_ERROR(ctx, log::message = "failed to parse get params response", log::exception = ex,
            log::where_name = MSETTINGS);
    }

    return {};
}

} // namespace NNotSoLiteSrv::NMSettings
