#include "util.h"

#include <mail/message_types/lib/message_types.h>
#include <mail/notsolitesrv/lib/mthr/src/message_id.h>
#include <mail/notsolitesrv/lib/mthr/src/util.h>
#include <mail/notsolitesrv/src/util/file.h>

#include <util/generic/string.h>
#include <util/string/cast.h>

#include <utility>

namespace NNotSoLiteSrv::NMthr {

using NUtil::ReadFile;
using ::NMthr::IsValidMessageId;
using ::NMthr::ReadMergeRulesFromJson;
using ::NMthr::ReadTrivialSubjectsFromJson;
using ::NMthr::SanitizeUTF8;
using ::NMthr::TMailThreadInfo;
using ::NMthr::TMergeRules;
using ::NMthr::TMessageInfo;

TMergeRules InitializeMergeRules(const NConfig::TMthr& config) {
    return {ReadMergeRulesFromJson(ReadFile(config.MergeRulesPath)),
        ReadTrivialSubjectsFromJson(ReadFile(config.TrivialSubjectsPath))};
}

void SetReferences(TMessageInfo& messageInfo, const std::vector<std::string>& references) {
    for (const auto& ref : references) {
        auto sanitize_ref = SanitizeUTF8(ref);
        if (IsValidMessageId(sanitize_ref)) {
            messageInfo.References.emplace_back(std::move(sanitize_ref));
        }
    }
}

TMessageInfo BuildMessageInfo(const TMthrRequest& request) {
    TMessageInfo messageInfo;
    messageInfo.HeaderFromDomain = SanitizeUTF8(request.HdrFromDomain);
    messageInfo.Subject = SanitizeUTF8(request.Subject);

    if (request.MessageId) {
        messageInfo.MessageId = SanitizeUTF8(*request.MessageId);
    }

    if (request.InReplyTo) {
        messageInfo.InReplyTo = SanitizeUTF8(*request.InReplyTo);
    }

    if (request.DomainLabel) {
        messageInfo.DomainLabel = std::move(*request.DomainLabel);
    }

    if (request.References && request.References->size()) {
        SetReferences(messageInfo, *request.References);
    }

    for (const auto typeId : request.MsgTypes) {
        messageInfo.MessageTypes.emplace(static_cast<NMail::EMessageType>(typeId));
    }

    return messageInfo;
}

TMailThreadInfo CalcDefaultThreadInfo(const TMessageInfo& messageInfo, const TMergeRules& mergeRules) {
    const auto defaultMergeRule = FindMergeRuleSafe(messageInfo, mergeRules);
    const auto defaultThreadInfo = CalcMailThreadInfo(messageInfo, defaultMergeRule);
    return defaultThreadInfo;
}

TThreadInfo MakeThreadInfo(const TMailThreadInfo& mailThreadInfo) {
    TThreadInfo threadInfo;
    threadInfo.Hash.Namespace = ToString(mailThreadInfo.Hash.NameSpace);
    threadInfo.Hash.Value = std::to_string(mailThreadInfo.Hash.Value);
    threadInfo.Rule = ToString(mailThreadInfo.MergePolicy);
    threadInfo.Limits.Days = mailThreadInfo.Limits.DaysLimit;
    threadInfo.Limits.Count = mailThreadInfo.Limits.CountLimit;
    threadInfo.InReplyToHash = std::to_string(mailThreadInfo.InReplyToHash);
    threadInfo.MessageIdHash = std::to_string(mailThreadInfo.MessageIdHash);

    threadInfo.ReferenceHashes.reserve(mailThreadInfo.ReferenceHashes.size());
    for (const auto hash : mailThreadInfo.ReferenceHashes) {
        threadInfo.ReferenceHashes.emplace_back(std::to_string(hash));
    }

    threadInfo.MessageIds.reserve(mailThreadInfo.MessageIds.size());
    for (const auto& id : mailThreadInfo.MessageIds) {
        threadInfo.MessageIds.emplace_back(std::move(id));
    }

    return threadInfo;
}

TMthrResponse MakeMthrResponse(const TMailThreadInfo& mailThreadInfo) {
    return {MakeThreadInfo(mailThreadInfo)};
}

}
