#include "multiput.h"
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <yplatform/coroutine.h>
#include <yplatform/future/multi_future.hpp>

namespace NNotSoLiteSrv::NMulcagate::NMulti {

namespace NDetail {

namespace NFuture = yplatform::future;

#include <yplatform/yield.h>
class TRequester {
public:
    using TYieldCtx = yplatform::yield_context<TRequester>;

    TRequester(
        TContextPtr ctx,
        const TRequests& request,
        TClientPtr mulcagateClient,
        TCallback callback
    )
        : Ctx(ctx)
        , Request(std::move(request))
        , MulcagateClient(mulcagateClient)
        , Callback(std::move(callback))
    {}

    void operator()(TYieldCtx yctx, TErrorCode ec = TErrorCode()) {
        reenter (yctx) {
            if (Request.empty()) {
                NSLS_LOG_CTX_ERROR(logdog::message="empty list of requests to mulcagate", logdog::where_name=Where);
                ErrorCode = ec = EError::DeliveryInternal;
                yield break;
            }

            NSLS_LOG_CTX_DEBUG(
                logdog::message="start process " + std::to_string(Request.size()) + " requests to mulcagate",
                logdog::where_name=Where);
            Processes.reserve(Request.size());
            std::transform(Request.begin(), Request.end(), std::back_inserter(Processes),
                [](const auto& req) { return TProcess(req.first, req.second); });
            Future = NFuture::future_multi_and(Processes, [](const auto& p) { return p.Future; });
            yield {
                Future.add_callback([this, yctx]() {
                    TErrorCode ec = EError::Ok;
                    try {
                        Future.get();
                    } catch (const std::exception& e) {
                        NSLS_LOG_CTX_ERROR(
                            logdog::message="mulcagate multiput error",
                            logdog::exception=e,
                            logdog::where_name=Where);
                        ec = EError::StorageError;
                    }
                    yctx(ec);
                });

                for (; CurProcIdx < Processes.size(); ++CurProcIdx) {
                    MulcagateClient->Put(
                        Ctx->GetTaskContext(Where),
                        Processes[CurProcIdx].MdsId,
                        Processes[CurProcIdx].Request.Data,
                        Processes[CurProcIdx].Request.Spam ? NMds::NS_SPAM : NMds::NS_MAIL,
                        [curProcIdx=CurProcIdx, this](TErrorCode ec, const std::string& stid) {
                            if (ec) {
                                NSLS_LOG_CTX_ERROR(
                                    logdog::message="mulcagate error",
                                    logdog::error_code=ec,
                                    logdog::where_name=Where);
                            } else {
                                NSLS_LOG_CTX_DEBUG(
                                    logdog::message="put successful, stid=" + stid + " , mulcaid=" + stid,
                                    logdog::where_name=Where);
                            }

                            auto& proc = Processes[curProcIdx];
                            proc.ErrorCode = ec;
                            proc.Stid = stid;
                            proc.Promise.set();
                        });
                }
                return;
            }

            if (ec) {
                ErrorCode = ec;
                yield break;
            }
        }

        if (yctx.is_complete()) {
            TResults result;
            std::transform(Processes.begin(), Processes.end(), std::inserter(result, result.end()),
                [](const auto& p) { return std::make_pair(p.Request.Uid, TResult{p.ErrorCode, p.Stid}); });
            Callback(ErrorCode, result);
        }
    }

private:
    TContextPtr Ctx;
    TRequests Request;
    TClientPtr MulcagateClient;
    TCallback Callback;
    const std::string Where {"MG"};

    struct TProcess {
        TProcess(const std::string& id, const TRequest& req)
            : MdsId(id)
            , Request(req)
        {
            Future = Promise;
        }

        NFuture::promise<void> Promise;
        NFuture::future<void> Future;
        std::string MdsId;
        const TRequest& Request;
        TErrorCode ErrorCode = EError::StorageError;
        std::string Stid;
    };
    std::vector<TProcess> Processes;
    size_t CurProcIdx = 0;
    NFuture::future<void> Future;
    TErrorCode ErrorCode;
};
#include <yplatform/unyield.h>

} // namespace NDetail

void Put(TContextPtr ctx, const TRequests& request, TClientPtr client, TCallback callback) {
    auto requester = std::make_shared<NDetail::TRequester>(ctx, std::move(request), client, std::move(callback));
    yplatform::spawn(requester);
}

} // namespace NNotSoLiteSrv::NMulcagate::NMulti
