#include "multiuser_processor.h"
#include <mail/notsolitesrv/src/user/processor.h>
#include <mail/notsolitesrv/src/tskv/logger.h>

#include <yplatform/coroutine.h>
#include <yplatform/future/multi_future.hpp>
#include <yplatform/log.h>

#include <boost/range/adaptor/transformed.hpp>

namespace NNotSoLiteSrv {

namespace NDetail {

#include <yplatform/yield.h>

namespace NFuture = yplatform::future;

class TMultiUserProcessor : public std::enable_shared_from_this<TMultiUserProcessor> {
public:
    using TYieldCtx = yplatform::yield_context<TMultiUserProcessor>;

    TMultiUserProcessor(
        TContextPtr ctx,
        NUser::TProcessor&& userProcessor,
        NUser::TStorage& userStorage,
        TMultiUserCallback&& cb
    )
        : Ctx(ctx)
        , ProcessUser(std::move(userProcessor))
        , Users(userStorage.GetUsers())
        , IsMailish(userStorage.IsMailish())
        , Callback(std::move(cb))
    {}

    void operator()(TYieldCtx yctx, TErrorCode ec = TErrorCode()) {
        reenter (yctx) {
            if (Users.empty()) {
                Result = EError::DeliveryNoRecipients;
                yield break;
            }

            yield {
                NSLS_LOG_CTX_DEBUG(
                    logdog::message="start process " + std::to_string(Users.size()) + " recipients",
                    logdog::where_name=Where);
                Processes.resize(Users.size());
                Future = NFuture::future_multi_and(Processes, [](const auto& p) { return p.Future; });
                Future.add_callback([this, yctx]() mutable {
                    try {
                        Future.get();
                    } catch (const std::exception& e) {
                        NSLS_LOG_CTX_ERROR(
                            logdog::message="multiuser processor exception",
                            logdog::exception=e,
                            logdog::where_name=Where);
                        Result = EError::MultiUserError;
                    }
                    yctx(Result);
                });

                size_t curRcptIdx = 0;
                using namespace boost::adaptors;
                auto range = Users | transformed([](auto& item) { return std::pair(&item.first, &item.second); });
                for (auto&& rcpt : std::vector(range.begin(), range.end())) {
                    ProcessUser(
                        Ctx,
                        NBlackbox::GetUser,
                        NBlackWhiteList::LoadLists,
                        *rcpt.first,
                        IsMailish,
                        !rcpt.second->DeliveryParams.NeedDelivery,
                        *rcpt.second,
                        [this, curRcptIdx](TErrorCode ec) {
                            auto& process = Processes[curRcptIdx];
                            if (ec) {
                                process.Result = ec;
                            }
                            process.Promise.set();
                        });

                    ++curRcptIdx;
                }
            }

            if (ec) {
                Result = ec;
                yield break;
            }
        }

        if (yctx.is_complete()) {
            Callback(Result);
        }
    }

private:
    TContextPtr Ctx;
    NUser::TProcessor ProcessUser;
    NUser::TStorage::TContainer& Users;
    bool IsMailish;
    TMultiUserCallback Callback;
    TErrorCode Result;
    const std::string Where {"MUP"};

    struct TUserFuture {
        TUserFuture() { Future = Promise; }

        NFuture::promise<void> Promise;
        NFuture::future<void> Future;
        TErrorCode Result;
    };
    std::vector<TUserFuture> Processes;
    NFuture::future<void> Future;
};

} // namespace NDetail

#include <yplatform/unyield.h>

void ProcessAllUsers(
    TContextPtr ctx,
    NUser::TProcessor userProcessor,
    NUser::TStorage& userStorage,
    TMultiUserCallback cb)
{
    auto processor = std::make_shared<NDetail::TMultiUserProcessor>(ctx, std::move(userProcessor), userStorage, std::move(cb));
    yplatform::spawn(processor);
}

} // namespace NNotSoLiteSrv
