#include "autoreplies.h"
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/util/headers.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/predicate.hpp>

namespace NNotSoLiteSrv::NNewEmails {

TAutoRepliesDataProvider::TAutoRepliesDataProvider(
    TContextPtr ctx,
    const TEmail& email,
    const NUser::TUser& user,
    TMessagePtr message,
    boost::asio::io_context& ioContext,
    NMSettings::TMSettingsClientPtr mSettingsClient
)
    : TDataProvider(ctx, email, user, message, "RPL")
    , IoContext(ioContext)
    , MSettingsClient(mSettingsClient)
{
}

const NConfig::TSmtpClient& TAutoRepliesDataProvider::GetConfig() const {
    return *Ctx->GetConfig()->AutoReply;
}

bool TAutoRepliesDataProvider::IsAcceptedMessage() const {
        if (!TDataProvider::IsAcceptedMessage()) {
            return false;
        }
        if (User.DeliveryResult.AutoReplies.empty()) {
            return false;
        }
        if (Message->IsSpam(User.Uid)) {
            NSLS_LOG_CTX_NOTICE(
                logdog::message="autoreply action skipped; for " + Email + ": SPAM",
                logdog::where_name=Where);
            return false;
        }
        if (Message->IsAutoReply() ||
            Message->IsAutoSubmitted() ||
            (boost::iequals(Message->GetPrecedence(), "bulk") && !Message->GetXYHintByUid(User.Uid).allow_autoreply))
        {
            NSLS_LOG_CTX_NOTICE(
                logdog::message="ommiting autoreply from " + Email + ": no replies to auto-generated messages",
                logdog::where_name=Where);
            return false;
        }
        return true;
}

void TAutoRepliesDataProvider::AsyncLoadExternalData(TDataProvider::TCallback callback) {
    auto resolver = std::make_shared<TSettingsGetter>(Ctx, User.Uid, MSettingsClient, [this, callback](TErrorCode ec, std::string fromName) {
        if (!fromName.empty()) {
            DisplayName = std::move(fromName);
        }
        callback(ec);
    });
    yplatform::spawn(IoContext.get_executor(), resolver);
}

std::string TAutoRepliesDataProvider::GetSenderName() const {
    return "autoreply";
}

TEmail TAutoRepliesDataProvider::GetMailFrom() const {
    if (!User.DefaultEmail.empty()) {
        return User.DefaultEmail;
    }

    return Email;
}

std::string TAutoRepliesDataProvider::GetFromDisplayName() const {
    std::string displayName;
    if (DisplayName) {
        displayName = *DisplayName;
    } else if (!User.FIO.empty()) {
        displayName = User.FIO;
    } else {
        return {};
    }

    return NUtil::EncodeRfc2047(displayName) + " ";
}

std::vector<TRecipient> TAutoRepliesDataProvider::GetRecipients() const {
    // auto-reply only with the first body
    return {{User.DeliveryResult.AutoReplies.front().first, {}, {}}};
}

std::string TAutoRepliesDataProvider::MakeReSubject() const {
    auto subj = Message->GetSubject();
    if (boost::istarts_with(subj, "re:")) {
        return NUtil::EncodeRfc2047(subj);
    } else {
        return NUtil::EncodeRfc2047("Re: " + subj);
    }
}

std::string TAutoRepliesDataProvider::GetMessage() const {
    std::ostringstream os;
    os << "Subject: " << MakeReSubject() << "\r\n";
    os << "To: <" << User.DeliveryResult.AutoReplies.front().first << ">\r\n";
    os << "Date: " << NUtil::MakeRfc2822Date(time(nullptr)) << "\r\n";
    os << "From: " << GetFromDisplayName() << "<" << GetMailFrom() << ">\r\n";
    os << "Message-Id: " << NUtil::MakeMessageId() << "\r\n";
    os << "MIME-Version: 1.0\r\n";
    os << "X-Mailer: Yamail [ http://yandex.ru ] 5.0\r\n";
    os << "X-Yandex-Forward: " << NUtil::MakeXYForwardValue(User.Suid) << "\r\n";
    os << "X-AutoReply: YES\r\n";
    os << "Auto-Submitted: auto-replied\r\n";
    if (!Message->GetMessageId().empty()) {
        os << "In-Reply-To: " << Message->GetMessageId() << "\r\n";
        os << "References: " << Message->GetMessageId();
        const auto& refs = Message->GetReferences();
        if (refs.empty()) {
            os << "\r\n";
        } else {
            os << " " << boost::join(refs, " ") << "\r\n";
        }
    }
    os << "Content-Type: text/plain; charset=\"utf-8\"\r\n";
    os << "Content-Transfer-Encoding: 8bit\r\n";
    os << "\r\n";
    os << User.DeliveryResult.AutoReplies.front().second << "\r\n";

    return os.str();
}

TDeliveryFlags TAutoRepliesDataProvider::GetDeliveryFlags() const {
    return {true, true, true};
}

#include <yplatform/yield.h>
void TAutoRepliesDataProvider::TSettingsGetter::operator()(
    TYieldCtx yctx, TErrorCode ec, const NMSettings::TParamsResult& result) const
{
    reenter (yctx) {
        yield MSettingsClient->GetProfile(Ctx, MakeRequest(), [yctx](TErrorCode ec, auto result) {
            return yctx(ec, result);
        });
    }

    if (yctx.is_complete()) {
        if (result.has_value() && result->FromName) {
            Callback(ec, *result->FromName);
        } else {
            Callback(ec, {});
        }
    }
}
#include <yplatform/unyield.h>

TDataProviderPtr CreateAutoRepliesDataProvider(
    TContextPtr ctx,
    const TEmail& email,
    const NUser::TUser& user,
    TMessagePtr msg,
    boost::asio::io_context& ioContext,
    NMSettings::TMSettingsClientPtr mSettingsClient)
{
    return TDataProviderPtr{new TAutoRepliesDataProvider(ctx, email, user, msg, ioContext, mSettingsClient)};
}

} // namespace NNotSoLiteSrv::NNewEmails
