#include "notifies.h"
#include <mail/notsolitesrv/src/config/smtpclient_notify.h>
#include <mail/notsolitesrv/src/util/headers.h>
#include <boost/algorithm/string/predicate.hpp>

namespace NNotSoLiteSrv::NNewEmails {

TNotifiesDataProvider::TNotifiesDataProvider(
    TContextPtr ctx,
    const TEmail& email,
    const NUser::TUser& user,
    TMessagePtr message
)
    : TDataProvider(ctx, email, user, message, "NTF")
{
}

const NConfig::TSmtpClient& TNotifiesDataProvider::GetConfig() const {
    return *Ctx->GetConfig()->Notify;
}

bool TNotifiesDataProvider::IsAcceptedMessage() const {
    if (!TDataProvider::IsAcceptedMessage()) {
        return false;
    }
    if (User.DeliveryResult.Notifies.empty()) {
        return false;
    }
    return true;
}

std::string TNotifiesDataProvider::GetSenderName() const {
    return "notify";
}

TEmail TNotifiesDataProvider::GetMailFrom() const {
    return Ctx->GetConfig()->Notify->MailFrom;
}

std::vector<TRecipient> TNotifiesDataProvider::GetRecipients() const {
    std::vector<TRecipient> ret;
    std::transform(
        User.DeliveryResult.Notifies.begin(),
        User.DeliveryResult.Notifies.end(),
        std::back_inserter(ret),
        [](const std::string& email) -> TRecipient {
            return {email, {}, {}};
        });
    return ret;
}

std::string TNotifiesDataProvider::GetMessage() const {
    std::ostringstream os;
    // TODO: get subject from tanker
    os << "Subject: =?utf-8?B?0J3QvtCy0L7QtSDQv9C40YHRjNC80L4g0LIg0K/QvdC00LXQutGBLtCf0L7Rh9GC0LUK?=\r\n";
    if (User.DeliveryResult.Notifies.size() == 1) {
        os << "To: <" << User.DeliveryResult.Notifies.front() << ">\r\n";
    } else {
        os << "To: undisclosed-recipients:;\r\n";
    }
    os << "Date: " << NUtil::MakeRfc2822Date(time(nullptr)) << "\r\n";
    if (Message->GetFrom()) {
        const auto& from = Message->GetFrom()->front();
        auto displayName = from.second; // TODO: review this, leaved for fastsrv java-tests compatibility
        os << "From: \"" << displayName << "\" <devnull@yandex.ru>\r\n";
    } else {
        // TODO: get default display name from tanker
        os << "From: =?utf-8?B?0K/QvdC00LXQutGBLtCf0L7Rh9GC0LAK?= <devnull@yandex.ru>\r\n";
    }
    os << "Message-Id: " << NUtil::MakeMessageId() << "\r\n";
    os << "MIME-Version: 1.0\r\n";
    os << "X-Mailer: Yamail [ http://yandex.ru ] 5.0\r\n";
    os << "X-AutoReply: YES\r\n";
    os << "Auto-Submitted: auto-generated\r\n";
    os << "Content-Type: text/plain; charset=\"utf-8\"\r\n";
    os << "Content-Transfer-Encoding: 8bit\r\n";

    // copy original mail X-Y-Forwards and add it for sender's suid (MAILDLV-3395)
    for (const auto& xyf: Message->GetXYForwardValues()) {
        os << "X-Yandex-Forward: " << xyf << "\r\n";
    }
    os << "X-Yandex-Forward: " << NUtil::MakeXYForwardValue(User.Suid) << "\r\n";

    os << "\r\n";
    // TODO: get message body from tanker
    os << "Вам письмо на " << Email << ". Прочитать: " << GetUnreadUrl() << " / Яндекс.Почта\r\n";

    return os.str();
}

std::string TNotifiesDataProvider::GetUnreadUrl() const {
    // TODO: Should be in the message body from tanker
    if (boost::iequals(User.Country, "ru")) {
        return "https://ya.cc/ZZZE";
    }
    if (boost::iequals(User.Country, "ua")) {
        return "https://ya.cc/ZZ_U";
    }
    if (boost::iequals(User.Country, "by")) {
        return "https://ya.cc/ZZ_i";
    }
    if (boost::iequals(User.Country, "kz")) {
        return "https://ya.cc/ZZ_u";
    }
    if (boost::iequals(User.Country, "tr")) {
        return "https://ya.cc/ZZZm";
    }
    return "https://ya.cc/ZZ_A";
}

TDeliveryFlags TNotifiesDataProvider::GetDeliveryFlags() const {
    return {false, false, true};
}

TDataProviderPtr CreateNotifiesDataProvider(
    TContextPtr ctx,
    const TEmail& email,
    const NUser::TUser& user,
    TMessagePtr msg)
{
    return TDataProviderPtr{new TNotifiesDataProvider(ctx, email, user, msg)};
}

} // namespace NNotSoLiteSrv::NNewEmails
