#include "processor.h"

#include "autoreplies.h"
#include "domain_rule_forwards.h"
#include "forwards.h"
#include "notifies.h"
#include "related_uids.h"

#include <mail/notsolitesrv/src/tskv/logger.h>

#include <exception>
#include <memory>
#include <utility>

namespace NNotSoLiteSrv::NNewEmails {

TProcessor::TProcessor(TContextPtr ctx, TAsyncSender sender, TDataProviderPtr dataProvider)
    : Ctx(ctx)
    , AsyncSend(std::move(sender))
    , Provider(std::move(dataProvider))
    , Where(Provider->GetWherePrefix())
{}

void TProcessor::Process(TCallback callback) {
    Callback = std::move(callback);
    yplatform::spawn(shared_from_this());
}

#include <yplatform/yield.h>
void TProcessor::operator()(TYieldCtx yctx, TErrorCode errc) {
    try {
        reenter (yctx) {
            yield Provider->AsyncLoadExternalData(yctx);
            if (errc) {
                yield break;
            }

            yield AsyncSend(Ctx, *Provider, yctx);
            if (errc) {
                yield break;
            }
            for (const auto& rcpt: Provider->GetRecipients()) {
                NSLS_LOG_CTX_NOTICE(
                    logdog::message=
                        Provider->GetSenderName() + " message submit from=<" + Provider->GetMailFrom() +
                        ">, to=<" + rcpt.Email + ">",
                    logdog::where_name=Where);
            }
        }

        if (yctx.is_complete()) {
            return Callback(errc);
        }
    } catch (const std::exception& e) {
        NSLS_LOG_CTX_ERROR(
            logdog::message="unexpected exception",
            logdog::exception=e,
            logdog::where_name=Where);
        return Callback(EError::DeliveryInternal);
    }
}
#include <yplatform/unyield.h>

TProcessors CreateSuitableProcessors(
    TContextPtr ctx,
    TProcessor::TAsyncSender sender,
    const NUser::TStorage& userStorage,
    TMessagePtr message,
    const TEmail& mailFrom,
    boost::asio::io_context& ioContext,
    NMSettings::TMSettingsClientPtr mSettingsClient)
{
    TProcessors ret;
    auto users = userStorage.GetFilteredUsers(NUser::Found) | NUser::ForDelivery |
        [](const auto& u) { return !u.DeliveryResult.IsDuplicate; };

    for (const auto& [email, user]: users) {
        auto majorProvider = CreateRelatedUidsDataProvider(ctx, email, user, message, mailFrom);
        if (majorProvider->IsAcceptedMessage()) {
            ret.emplace_back(std::make_unique<TProcessor>(ctx, sender, std::move(majorProvider)));
        }

        if (!user.DeliveryResult.DomainRuleForwards.empty()) {
            auto provider{CreateDomainRuleForwardsDataProvider(ctx, email, user, message)};
            if (provider->IsAcceptedMessage()) {
                ret.emplace_back(std::make_unique<TProcessor>(ctx, sender, std::move(provider)));
            }
        }

        if (!user.DeliveryResult.Forwards.empty()) {
            auto provider = CreateForwardsDataProvider(ctx, email, user, message);
            if (provider->IsAcceptedMessage()) {
                ret.emplace_back(std::make_unique<TProcessor>(ctx, sender, std::move(provider)));
            }
        }

        if (!user.DeliveryResult.Notifies.empty()) {
            auto provider = CreateNotifiesDataProvider(ctx, email, user, message);
            if (provider->IsAcceptedMessage()) {
                ret.emplace_back(std::make_unique<TProcessor>(ctx, sender, std::move(provider)));
            }
        }

        if (!user.DeliveryResult.AutoReplies.empty()) {
            auto provider = CreateAutoRepliesDataProvider(ctx, email, user, message, ioContext, mSettingsClient);
            if (provider->IsAcceptedMessage()) {
                ret.emplace_back(std::make_unique<TProcessor>(ctx, sender, std::move(provider)));
            }
        }
    }

    return ret;
}

} // namespace NNotSoLiteSrv::NNewEmails
