#pragma once

#include <yamail/data/reflection.h>
#include <boost/lexical_cast.hpp>
#include <util/string/cast.h>
#include <map>
#include <string>

namespace NNotSoLiteSrv::NReflection {

using TStringMultiMap = std::multimap<std::string, std::string>;

struct TStartTag {};

class TStringMultiMapReader: public yreflection::Visitor {
private:
    using TRange = std::pair<TStringMultiMap::const_iterator, TStringMultiMap::const_iterator>;

public:
    explicit TStringMultiMapReader(const TStringMultiMap& mmap)
        : Mmap(mmap)
        , Range(mmap.begin(), mmap.end())
    {}
    TStringMultiMapReader(const TStringMultiMap& mmap, TRange range): Mmap(mmap), Range(range) {}

    template <typename T>
    void Apply(T& res) {
        applyVisitor(res, *this, TStartTag());
    }

    template <typename TValue, typename ... TArgs, std::enable_if_t<!std::is_same_v<TValue, bool>>* TType = nullptr>
    void onValue(TValue& v, yreflection::NamedItemTag<TArgs...> tag) const {
        auto it = Mmap.find(yreflection::name(tag));
        if (it != Mmap.end()) {
            try {
                v = boost::lexical_cast<TValue>(it->second);
            } catch (...) {
            }
        }
    }

    template <typename TValue, typename ... TArgs, std::enable_if_t<std::is_same_v<TValue, bool>>* TType = nullptr>
    void onValue(TValue& v, yreflection::NamedItemTag<TArgs...> tag) const {
        auto it = Mmap.find(yreflection::name(tag));
        if (it != Mmap.end()) {
            int intV;
            TryFromStringWithDefault<int>(TStringBuf{it->second}, intV, 0);
            v = intV;
        }
    }

    template <typename TValue>
    void onValue(TValue& v, yreflection::SequenceItemTag) {
        if (Range.first == Range.second) {
            throw std::runtime_error("incorrect sequence size!");
        }

        try {
            v = boost::lexical_cast<TValue>(Range.first->second);
        } catch (...) {
        }
        ++Range.first;
    }

    template <typename TStruct, typename TTag>
    TStringMultiMapReader onStructStart(TStruct&, TTag) const {
        throw std::runtime_error("structs are impossible in multimaps");
    }

    template <typename TStruct>
    TStringMultiMapReader onStructStart(TStruct&, TStartTag) {
        return *this;
    }

    template <typename TStruct, typename TTag>
    void onStructEnd(TStruct&, TTag) const {
    }

    template <typename TSequence, typename TTag>
    TStringMultiMapReader onSequenceStart(TSequence&, TTag) const {
        throw std::runtime_error("nameless sequences are impossible in multimaps");
    }

    template <typename TSequence, typename ... TArgs>
    TStringMultiMapReader onSequenceStart(TSequence& s, yreflection::NamedItemTag<TArgs...> tag) {
        auto erange = Mmap.equal_range(yreflection::name(tag));
        s.resize(std::distance(erange.first, erange.second));

        return TStringMultiMapReader(Mmap, erange);
    }

    template <typename TSequence, typename TTag>
    void onSequenceEnd(TSequence&, TTag) const {
    }

    template <typename TPointer, typename ... TArgs>
    bool onSmartPointer(TPointer& p, yreflection::NamedItemTag<TArgs...> tag) const {
        bool found = (Mmap.find(yreflection::name(tag)) != Mmap.end());
        if (found) {
            p.reset(new typename TPointer::element_type);
        }
        return found;
    }

    template <typename TPointer>
    bool onSmartPointer(TPointer& p, TStartTag) const {
        p.reset(new typename TPointer::element_type);
        return true;
    }

    template <typename TPointer>
    bool onSmartPointer(TPointer& p, yreflection::SequenceItemTag) const {
        bool found = (Range.first != Range.second);
        if (found) {
            p.reset(new typename TPointer::element_type);
        }

        return found;
    }

    template <typename TInnerType, typename ... TArgs>
    bool onOptional(boost::optional<TInnerType>& p, yreflection::NamedItemTag<TArgs...> tag) const {
        auto it = Mmap.find(yreflection::name(tag));
        if (it != Mmap.end()) {
            p = boost::lexical_cast<TInnerType>(it->second);
        }
        return p.is_initialized();
    }

private:
    const TStringMultiMap& Mmap;
    TRange Range;
};

template <typename TResult>
inline TResult FromStringMultiMap(const TStringMultiMap& mmap) {
    TResult res;
    TStringMultiMapReader(mmap).Apply(res);
    return res;
}

} // namespace NNotSoLiteSrv::NReflection
