#include "xyandexhint.h"
#include "multimap.h"
#include <mail/library/utf8/utf8.h>
#include <butil/butil.h>
#include <yamail/data/reflection.h>
#include <boost/algorithm/string/trim.hpp>
#include <boost/mpl/range_c.hpp>
#include <boost/tokenizer.hpp>
#include <util/generic/algorithm.h>

BOOST_FUSION_ADAPT_STRUCT(NNotSoLiteSrv::TXYandexHint,
    service,
    hdr_date,
    received_date,
    folder,
    folder_path,
    folder_spam_path,
    folder_path_delim,
    fid,
    mixed,
    skip_loop_prevention,
    skip_loop_prevention_for_suid,
    ipfrom,
    method_id,
    session_key,
    notify,
    forward,
    filters,
    imap,
    label,
    imaplabel,
    userlabel,
    lid,
    bcc,
    source_stid,
    mid,
    email,
    copy_to_inbox,
    save_to_sent,
    final_headers_len,
    body_md5,
    meta_blob,
    external_imap_id,
    allow_autoreply,
    disable_push,
    priority_high,
    phone,
    session_id,
    store_as_deleted,
    skip_meta_msg,
    sync_dlv,
    no_such_fid_fail,
    replace_so_labels
);

namespace NNotSoLiteSrv::NReflection {

template <typename TIter>
TXYandexHint ParseXYHint(TIter begin, TIter end) {
    boost::char_separator<char> sep("\n");
    std::string tmpStr(begin, end);
    boost::tokenizer<boost::char_separator<char>> tokens(tmpStr, sep);
    TStringMultiMap mmap;

    for (const auto& tok: tokens) {
        auto token = boost::trim_copy(tok);
        auto eqPos = token.find('=');
        if (eqPos == std::string::npos) {
            continue;
        }

        auto key = token.substr(0, eqPos);
        boost::trim(key);
        auto val = NUtil::Utf8Sanitized(token.substr(eqPos + 1));
        boost::trim(val);

        mmap.insert({key, val});
    }

    return FromStringMultiMap<TXYandexHint>(mmap);
}

TXYandexHint ParseXYHint(const std::string& base64XYHint) {
    auto hint = decode_base64(base64XYHint);
    return ParseXYHint(hint.begin(), hint.end());
}

TXYandexHint ParseHttpHint(const NHttp::THint& hint) {
    TStringMultiMap mmap;
    for (const auto& [key, values] : hint) {
        for (const auto& value : values) {
            mmap.emplace(key, value);
        }
    }

    return FromStringMultiMap<TXYandexHint>(mmap);
}

namespace NYDR = yamail::data::reflection;

template <typename T>
struct TIsSequence: std::bool_constant<NYDR::is_std_begin_end_compatible_range_v<T> && !NYDR::is_string_v<T>> {
};

template <typename T>
constexpr auto IsSequenceValue = TIsSequence<T>::value;

template <typename T, typename std::enable_if_t<!IsSequenceValue<T>>* = nullptr>
void Combine(T& to, const T& from, const T& def) {
    if (to == def) {
        to = from;
    }
}

template <typename T, typename std::enable_if_t<IsSequenceValue<T>>* = nullptr>
void Combine(T& to, const T& from, const T&) {
    for (auto from_val: from) {
        to.emplace_back(from_val);
    }
    SortUnique(to);
}

void CombineXYHint(TXYandexHint& to, const TXYandexHint& from) {
    TXYandexHint defaultValues{};
    using XYH = TXYandexHint;
    using zip_vec = boost::fusion::vector<XYH&, const XYH&, const XYH&>;
    boost::fusion::zip_view<zip_vec> zip_view(zip_vec(to, from, defaultValues));

    boost::fusion::for_each(zip_view, [](auto&& zip) {
        Combine(boost::fusion::at_c<0>(zip), boost::fusion::at_c<1>(zip), boost::fusion::at_c<2>(zip));
    });
}

struct TStringifier {
    TStringifier(const TXYandexHint& hint, std::string delim = " "): Hint(hint), Delimiter(delim) {}
    std::string Result() const { return Stream.str(); }

    template <typename Index>
    void operator()(Index idx) const {
        auto val = boost::fusion::at_c<idx>(Hint);
        if (val == boost::fusion::at_c<idx>(DefaultHint)) {
            return;
        }

        auto name = boost::fusion::extension::struct_member_name<TXYandexHint, idx>::call();
        Out(name, val);
    }

    template <typename T, typename std::enable_if_t<NYDR::is_optional_v<T>>* = nullptr>
    void Out(const std::string& name, const T& val) const {
        if (val) {
            Stream << name << "=" << *val << Delimiter;
        }
    }

    template <typename T, typename std::enable_if_t<!NYDR::is_optional_v<T> && IsSequenceValue<T>>* = nullptr>
    void Out(const std::string& name, const T& seq) const {
        for (const auto& val: seq) {
            Stream << name << "=" << val << Delimiter;
        }
    }

    template <typename T, typename std::enable_if_t<!NYDR::is_optional_v<T> && !IsSequenceValue<T>>* = nullptr>
    void Out(const std::string& name, const T& val) const {
        Stream << name << "=" << val << Delimiter;
    }

private:
    const TXYandexHint& Hint;
    std::string Delimiter;
    TXYandexHint DefaultHint;
    mutable std::ostringstream Stream;
};

std::string XYHintToString(const TXYandexHint& hint) {
    TStringifier stringifier{hint};
    using Indices = boost::mpl::range_c<unsigned, 0, boost::fusion::result_of::size<TXYandexHint>::value>;
    boost::fusion::for_each(Indices(), stringifier);
    return boost::trim_right_copy(stringifier.Result());
}

std::string MakeXYHintHeaderValue(const TXYandexHint& hint) {
    TStringifier stringifier{hint, "\n"};
    using Indices = boost::mpl::range_c<unsigned, 0, boost::fusion::result_of::size<TXYandexHint>::value>;
    boost::fusion::for_each(Indices(), stringifier);
    return encode_base64(stringifier.Result());
}

} // namespace NNotSoLiteSrv::NReflection
