#include "domain_rules.h"

#include <mail/notsolitesrv/src/rules/domain/util/common.h>
#include <mail/notsolitesrv/src/rules/domain/util/furita.h>
#include <mail/notsolitesrv/src/rules/domain/util/tupita.h>

#include <boost/range/adaptor/filtered.hpp>

#include <string>
#include <utility>
#include <variant>
#include <vector>

namespace NNotSoLiteSrv::NRules {

const std::string DOMAIN_RULES{"DOMAIN_RULES"};

TDomainRules::TDomainRules(TDomainRulesClients clients, TContextPtr ctx, NUser::TStoragePtr userStorage,
    boost::asio::io_context& ioContext)
    : Clients{std::move(clients)}
    , Ctx{std::move(ctx)}
    , UserStorage{std::move(userStorage)}
    , IoContext{ioContext}
{
}

void TDomainRules::SetParams(TMessagePtr message, NMetaSaveOp::TRequest request,
    TDomainRulesCallback callback)
{
    Params.Message = std::move(message);
    Params.Request = std::move(request);
    Params.Callback = std::move(callback);
}

#include <yplatform/yield.h>

void TDomainRules::operator()(TYieldCtx yieldCtx, TErrorCode errorCode, TDomainRules::TResult result) {
    reenter(yieldCtx) {
        if (!DomainRulesInUse(Params.Request.recipients)) {
            NSLS_LOG_CTX_NOTICE(log::message = "domain rules not in use", log::where_name = DOMAIN_RULES);
            return Params.Callback(std::move(errorCode));
        }

        FuritaOrgIds = MakeFuritaOrgIds(Params.Request.recipients);
        while (!FuritaOrgIds.empty()) {
            OrgId = std::move(FuritaOrgIds.extract(FuritaOrgIds.cbegin()).value());
            yield RequestFurita(yieldCtx);
            ProcessFuritaGetResult(std::move(errorCode), std::move(result));
            if (FuritaDomainRulesAvailable(ClientResults.Furita[OrgId])) {
                yield RequestTupita(yieldCtx);
                ProcessTupitaCheckResult(std::move(errorCode), std::move(result));
            }
        }
    }

    if (yieldCtx.is_complete()) {
        ProcessResponses();
    }
}

#include <yplatform/unyield.h>

void TDomainRules::RequestFurita(const TYieldCtx& yieldCtx) {
    Clients.Furita->Get(IoContext, OrgId, yieldCtx);
}

void TDomainRules::RequestTupita(const TYieldCtx& yieldCtx) {
    const auto uid{std::stoull(OrgId)};
    auto checkRequest{MakeTupitaCheckRequest(Params.Request, uid, ClientResults.Furita[OrgId].Result->Rules)};
    Clients.Tupita->Check(IoContext, uid, Params.Request.session_id, std::move(checkRequest), yieldCtx);
}

void TDomainRules::ProcessFuritaGetResult(TErrorCode errorCode, TResult result) {
    if (errorCode) {
        if (errorCode == EError::FuritaOrgNotFound) {
            ClientResults.Furita[OrgId] = MakeEmptyFuritaClientResult();
            NSLS_LOG_CTX_ERROR(log::message = "organization not found", log::org_id = OrgId,
                log::error_code = errorCode, log::where_name = DOMAIN_RULES);
        } else {
            ClientResults.Furita[OrgId].ErrorCode = errorCode;
            NSLS_LOG_CTX_ERROR(log::message = "failed to call furita", log::org_id = OrgId,
                log::error_code = ClientResults.Furita[OrgId].ErrorCode, log::where_name = DOMAIN_RULES);
        }
    } else if (!CheckSpecificResult<NFurita::TGetResult>(result)) {
        ClientResults.Furita[OrgId].ErrorCode = EError::DomainRulesIncorrectResult;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get furita result", log::org_id = OrgId,
            log::error_code = ClientResults.Furita[OrgId].ErrorCode, log::where_name = DOMAIN_RULES);
    } else {
        ClientResults.Furita[OrgId].Result = std::get<NFurita::TGetResult>(std::move(result));
    }
}

void TDomainRules::ProcessTupitaCheckResult(TErrorCode errorCode, TResult result) {
    if (errorCode) {
        ClientResults.Tupita[OrgId].ErrorCode = errorCode;
        NSLS_LOG_CTX_ERROR(log::message = "failed to call tupita", log::org_id = OrgId,
            log::error_code = ClientResults.Tupita[OrgId].ErrorCode, log::where_name = DOMAIN_RULES);
    } else if (!CheckSpecificResult<NTupita::TCheckResult>(result)) {
        ClientResults.Tupita[OrgId].ErrorCode = EError::DomainRulesIncorrectResult;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get tupita result", log::org_id = OrgId,
            log::error_code = ClientResults.Tupita[OrgId].ErrorCode, log::where_name = DOMAIN_RULES);
    } else {
        ClientResults.Tupita[OrgId].Result = std::get<NTupita::TCheckResult>(std::move(result));
        if (ClientResults.Tupita[OrgId].Result->Result.size() != 1) {
            NSLS_LOG_CTX_ERROR(log::message = "invalid tupita result", log::org_id = OrgId,
                log::where_name = DOMAIN_RULES);
        }
    }
}

void TDomainRules::ProcessResponses() const {
    ProcessErrors();
    ProcessSuccessfulResponses();
    Params.Callback({});
}

void TDomainRules::ProcessErrors() const {
    for (auto& element : UserStorage->GetFilteredUsers(MakeUserFilter())) {
        const auto orgId{*element.second.OrgId};
        if (ClientResults.Furita.at(orgId).ErrorCode) {
            element.second.DeliveryResult.ErrorCode = ClientResults.Furita.at(orgId).ErrorCode;
        } else if (ClientResults.Tupita.contains(orgId) && ClientResults.Tupita.at(orgId).ErrorCode) {
            element.second.DeliveryResult.ErrorCode = ClientResults.Tupita.at(orgId).ErrorCode;
        }
    }
}

void TDomainRules::ProcessSuccessfulResponses() const {
    auto filter{[](const auto& element){return (element.second.Result.has_value() &&
        (element.second.Result->Result.size() == 1));}};
    for (const auto& [orgId, tupitaClientResult] : ClientResults.Tupita | boost::adaptors::filtered(
        std::move(filter)))
    {
        ProcessTupitaUserWithMatchedQueries(orgId, tupitaClientResult.Result->Result[0]);
    }
}

void TDomainRules::ProcessTupitaUserWithMatchedQueries(const TOrgId& orgId,
    const NTupita::TTupitaUserWithMatchedQueries& tupitaUser) const
{
    if (!TupitaUserCorrect(orgId, tupitaUser)) {
        return;
    }

    ProcessFuritaDomainRules(orgId, MakeMatchedDomainRuleIndices(tupitaUser.MatchedQueries));
}

bool TDomainRules::TupitaUserCorrect(const TOrgId& orgId,
    const NTupita::TTupitaUserWithMatchedQueries& tupitaUser) const
{
    if (!MatchedQueriesNumeric(tupitaUser.MatchedQueries)) {
        NSLS_LOG_CTX_ERROR(log::message = "incorrect matched queries", log::org_id = orgId,
            log::where_name = DOMAIN_RULES);
        return false;
    }

    return true;
}

void TDomainRules::ProcessFuritaDomainRules(const TOrgId& orgId,
    const TMatchedDomainRuleIndices matchedRuleIndices) const
{
    const auto& furitaResponse{*ClientResults.Furita.at(orgId).Result};
    NSLS_LOG_CTX_NOTICE(log::message = "domain rules processing (revision = " + std::to_string(
        furitaResponse.Revision) + ")", log::org_id = orgId, log::where_name = DOMAIN_RULES);

    if (!MatchedDomainRuleIndicesAndRulesCorrect(orgId, furitaResponse.Rules, matchedRuleIndices)) {
        return;
    }

    ApplyDomainRulesAccumulatedResult(orgId, MakeDomainRulesAccumulatedResult(furitaResponse.Rules,
        matchedRuleIndices));
}

bool TDomainRules::MatchedDomainRuleIndicesAndRulesCorrect(
    const TOrgId& orgId,
    const std::vector<NFurita::TFuritaDomainRule>& rules,
    const TMatchedDomainRuleIndices& matchedRuleIndices) const
{
    if (!MatchedDomainRuleIndicesCorrect(rules.size(), matchedRuleIndices)) {
        NSLS_LOG_CTX_ERROR(log::message = "incorrect matched rule indices", log::org_id = orgId,
            log::where_name = DOMAIN_RULES);
        return false;
    }

    if (!MatchedDomainRulesCorrect(rules, matchedRuleIndices)) {
        NSLS_LOG_CTX_ERROR(log::message = "incorrect matched rules", log::org_id = orgId,
            log::where_name = DOMAIN_RULES);
        return false;
    }

    return true;
}

void TDomainRules::ApplyDomainRulesAccumulatedResult(const TOrgId& orgId,
    const TDomainRulesAccumulatedResult accumulatedResult) const
{
    auto orgIdFilter{[&](const auto& user){return (user.OrgId == orgId);}};
    for (auto& element : UserStorage->GetFilteredUsers(MakeUserFilter()) | std::move(orgIdFilter)) {
        element.second.DeliveryResult.DomainRuleIds = accumulatedResult.AppliedDomainRuleIds;
        element.second.DeliveryResult.DomainRuleForwards = accumulatedResult.Forwards;
        if (accumulatedResult.Drop) {
            Params.Message->StoreAsDeleted(element.second.Uid);
        }
    }
}

}
