#pragma once

#include "idomain_rules.h"

#include <mail/notsolitesrv/src/context.h>
#include <mail/notsolitesrv/src/furita/types/response.h>
#include <mail/notsolitesrv/src/rules/domain/types/furita.h>
#include <mail/notsolitesrv/src/rules/domain/types/tupita.h>
#include <mail/notsolitesrv/src/tupita/types/response.h>
#include <mail/notsolitesrv/src/types/common.h>
#include <mail/notsolitesrv/src/user/storage.h>

#include <boost/asio.hpp>

#include <map>
#include <set>
#include <variant>
#include <vector>

namespace NNotSoLiteSrv::NRules {

struct TDomainRulesClients {
    NFurita::TFuritaClientPtr Furita;
    NTupita::TTupitaClientPtr Tupita;
};

struct TDomainRulesParams {
    TMessagePtr Message;
    NMetaSaveOp::TRequest Request;
    TDomainRulesCallback Callback;
};

struct TFuritaClientResult {
    TErrorCode ErrorCode;
    NFurita::TGetResult Result;
};

using TFuritaClientResults = std::map<TOrgId, TFuritaClientResult>;

struct TTupitaClientResult {
    TErrorCode ErrorCode;
    NTupita::TCheckResult Result;
};

using TTupitaClientResults = std::map<TOrgId, TTupitaClientResult>;

struct TDomainRulesClientResults {
    TFuritaClientResults Furita;
    TTupitaClientResults Tupita;
};

class TDomainRules : public IDomainRules {
public:
    TDomainRules(TDomainRulesClients clients, TContextPtr ctx, NUser::TStoragePtr userStorage,
        boost::asio::io_context& ioContext);

    void SetParams(TMessagePtr message, NMetaSaveOp::TRequest request,
        TDomainRulesCallback callback) override;
    void operator()(TYieldCtx yieldCtx, TErrorCode errorCode = {}, TResult result = {}) override;

private:
    void RequestFurita(const TYieldCtx& yieldCtx);
    void RequestTupita(const TYieldCtx& yieldCtx);

    void ProcessFuritaGetResult(TErrorCode errorCode, TResult result);
    void ProcessTupitaCheckResult(TErrorCode errorCode, TResult result);

    void ProcessResponses() const;
    void ProcessErrors() const;
    void ProcessSuccessfulResponses() const;

    void ProcessTupitaUserWithMatchedQueries(const TOrgId& orgId,
        const NTupita::TTupitaUserWithMatchedQueries& tupitaUser) const;

    bool TupitaUserCorrect(const TOrgId& orgId,
        const NTupita::TTupitaUserWithMatchedQueries& tupitaUser) const;

    void ProcessFuritaDomainRules(const TOrgId& orgId,
        const TMatchedDomainRuleIndices matchedRuleIndices) const;

    bool MatchedDomainRuleIndicesAndRulesCorrect(
        const TOrgId& orgId,
        const std::vector<NFurita::TFuritaDomainRule>& rules,
        const TMatchedDomainRuleIndices& matchedRuleIndices) const;

    void ApplyDomainRulesAccumulatedResult(const TOrgId& orgId,
        const TDomainRulesAccumulatedResult accumulatedResult) const;

    template<typename TSpecificResult> bool CheckSpecificResult(const TResult& result) const {
        return std::holds_alternative<TSpecificResult>(result) && std::get<TSpecificResult>(result);
    }

    const TDomainRulesClients Clients;
    const TContextPtr Ctx;
    const NUser::TStoragePtr UserStorage;
    boost::asio::io_context& IoContext;
    TDomainRulesParams Params;
    std::set<TOrgId> FuritaOrgIds;
    TOrgId OrgId;
    TDomainRulesClientResults ClientResults;
};

}
