#include "furita.h"

#include <boost/range/adaptor/filtered.hpp>

#include <algorithm>
#include <set>
#include <string>
#include <unordered_set>
#include <utility>
#include <vector>

namespace NNotSoLiteSrv::NRules {

namespace {

const std::string DOMAIN_ACTION_DROP{"drop"};
const std::string DOMAIN_ACTION_FORWARD{"forward"};

}

std::set<TOrgId> MakeFuritaOrgIds(const NMetaSaveOp::TRecipientMap& recipients) {
    std::set<TOrgId> orgIds;
    auto filter{[](const auto& element){return element.second.params.use_domain_rules;}};
    for (const auto& element : recipients | boost::adaptors::filtered(std::move(filter))) {
        orgIds.emplace(*element.second.user.org_id);
    }

    return orgIds;
}

bool FuritaDomainRulesAvailable(const TFuritaClientResult& furitaClientResult) {
    return furitaClientResult.Result && (!furitaClientResult.Result->Rules.empty());
}

TFuritaClientResult MakeEmptyFuritaClientResult() {
    return {.ErrorCode = EError::Ok, .Result{{.Rules{}, .Revision = 0}}};
}

bool MatchedDomainRulesCorrect(const std::vector<NFurita::TFuritaDomainRule>& rules,
    const TMatchedDomainRuleIndices& matchedRuleIndices)
{
    for (const auto& index : matchedRuleIndices) {
        if (!DomainActionsCorrect(rules[index].Actions)) {
            return false;
        }
    }

    return true;
}

bool DomainActionsCorrect(const std::vector<NFurita::TFuritaDomainAction>& actions) {
    auto predicate{[&](const auto& action){return DomainActionCorrect(action);}};
    return std::all_of(actions.cbegin(), actions.cend(), std::move(predicate));
}

bool DomainActionCorrect(const NFurita::TFuritaDomainAction& action) {
    return (DomainActionActionCorrect(action.Action) && DomainActionDataCorrect(action.Action, action.Data));
}

bool DomainActionActionCorrect(const std::string& action) {
    const static std::unordered_set<std::string> supportedActions{DOMAIN_ACTION_DROP, DOMAIN_ACTION_FORWARD};
    return supportedActions.contains(action);
}

bool DomainActionDataCorrect(const std::string& action,
    const std::optional<NFurita::TFuritaDomainActionData>& data)
{
    if (action == DOMAIN_ACTION_FORWARD) {
        return (data && data->Email && (!data->Email->empty()));
    }

    return true;
}

TDomainRulesAccumulatedResult MakeDomainRulesAccumulatedResult(
    const std::vector<NFurita::TFuritaDomainRule>& rules,
    const TMatchedDomainRuleIndices& matchedRuleIndices)
{
    TDomainRulesAccumulatedResult result;
    for (const auto& index : matchedRuleIndices) {
        for (const auto& action : rules[index].Actions) {
            ApplyDomainAction(action, result);
        }

        result.AppliedDomainRuleIds.emplace_back(std::to_string(index));
        if (rules[index].Terminal.value_or(false)) {
            break;
        }
    }

    return result;
}

void ApplyDomainAction(const NFurita::TFuritaDomainAction& action, TDomainRulesAccumulatedResult& result) {
    if (action.Action == DOMAIN_ACTION_FORWARD) {
        result.Forwards.emplace_back(*action.Data->Email);
    } else if (action.Action == DOMAIN_ACTION_DROP) {
        result.Drop = true;
    }
}

}
