#include "rules_applier.h"

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/transform.hpp>

#include <algorithm>
#include <iterator>

namespace NNotSoLiteSrv::NRulesApplier {

void TApplyResult::Accumulate(const TApplyResult& other) {
    appliedWithStore |= other.appliedWithStore;
    appliedWithoutStore |= other.appliedWithoutStore;
}

TApplyResult::operator bool() const {
    return appliedWithStore || appliedWithoutStore;
}

TRulesApplierResponse ApplyRules(
    TRulesApplierRequest request,
    const NMetaSaveOp::TFuritaClientResults& furitaClientResults,
    const NMetaSaveOp::TTupitaClientResults& tupitaClientResults)
{
    auto filter{[](const auto& element){return element.second.UseFilters;}};
    const auto matchedRulesByUid{MakeMatchedRulesByUid(furitaClientResults, MakeMatchedRuleIdsByUid(
        tupitaClientResults))};
    TRulesApplierResponse response;
    for (auto& [recipientId, recipient] : request.Recipients | boost::adaptors::filtered(std::move(filter))) {
        if (FailedRecipient(recipient.Uid, furitaClientResults, tupitaClientResults)) {
            response.FailedRecipients.emplace_back(std::move(recipientId));
        } else if (matchedRulesByUid.contains(recipient.Uid)) {
            auto appliedRules{MakeAppliedRules(request.Message, matchedRulesByUid.at(recipient.Uid))};
            response.AppliedRules[std::move(recipientId)] = std::move(appliedRules);
        } else {
            response.AppliedRules.emplace(std::move(recipientId), TAppliedRules{});
        }
    }

    return response;
}

bool FailedRecipient(TUid uid, const NMetaSaveOp::TFuritaClientResults& furitaClientResults,
    const NMetaSaveOp::TTupitaClientResults& tupitaClientResults)
{
    return (furitaClientResults.at(uid).ErrorCode || (tupitaClientResults.contains(uid) &&
        tupitaClientResults.at(uid).ErrorCode));
}

TMatchedRuleIdsByUid MakeMatchedRuleIdsByUid(const NMetaSaveOp::TTupitaClientResults& tupitaClientResults) {
    auto filter{[](const auto& element){return (element.second.Result.has_value() &&
        (element.second.Result->Result.size() == 1));}};
    TMatchedRuleIdsByUid result;
    auto transformer{[](const auto& element) {
        const auto& matchedQueries{element.second.Result->Result[0].MatchedQueries};
        TMatchedRuleIds matchedRuleIds{matchedQueries.cbegin(), matchedQueries.cend()};
        return std::make_pair(element.first, std::move(matchedRuleIds));
    }};

    boost::transform(tupitaClientResults | boost::adaptors::filtered(std::move(filter)),
        std::inserter(result, result.end()), std::move(transformer));
    return result;
}

TMatchedRulesByUid MakeMatchedRulesByUid(const NMetaSaveOp::TFuritaClientResults& furitaClientResults,
    TMatchedRuleIdsByUid matchedRuleIdsByUid)
{
    TMatchedRulesByUid result;
    for (const auto& matchedRuleIds : matchedRuleIdsByUid) {
        auto rules{furitaClientResults.at(matchedRuleIds.first).Result->Rules};
        std::sort(rules.begin(), rules.end(), [](const auto& left, const auto& right){
            return left.Priority < right.Priority;});
        auto filter{[&](const auto& rule){return matchedRuleIds.second.contains(rule.Id);}};
        TMatchedRules matchedRules;
        for (auto& rule : rules | boost::adaptors::filtered(std::move(filter))) {
            matchedRules.emplace_back(std::move(rule));
            if (matchedRules.back().Stop) {
                break;
            }
        }

        result.emplace(matchedRuleIds.first, std::move(matchedRules));
    }

    return result;
}

TAppliedRules MakeAppliedRules(const TMessage& message, const TMatchedRules& matchedRules) {
    TApplyResult rulesApplyResult;
    TAppliedRules appliedRules;
    for (const auto& rule : matchedRules) {
        rulesApplyResult.Accumulate(ApplyRule(message, rule, appliedRules));
    }

    if (rulesApplyResult.appliedWithoutStore && !rulesApplyResult.appliedWithStore) {
        appliedRules.DestFolder.Path = "\\inbox";
        appliedRules.StoreAsDeleted = true;
    }

    return appliedRules;
}

TApplyResult ApplyRule(const TMessage& message, const NFurita::TFuritaRule& rule,
    TAppliedRules& appliedRules)
{
    TApplyResult ruleApplyResult;
    for (const auto& action : rule.Actions) {
        ruleApplyResult.Accumulate(ProcessApplyActionResult(ApplyAction(message, action, appliedRules)));
    }

    if (ruleApplyResult) {
        appliedRules.RuleIds.emplace_back(rule.Id);
    }

    return ruleApplyResult;
}

std::optional<EApplyResult> ApplyAction(const TMessage& message, const NFurita::TFuritaAction& action,
    TAppliedRules& appliedRules)
{
    if (!action.Verified) {
        return {};
    }

    auto& fid = appliedRules.DestFolder.Fid;
    auto& path = appliedRules.DestFolder.Path;
    const auto parameter = action.Parameter.value_or(std::string{});
    auto applyResult = EApplyResult::WithStore;
    if (action.Type == "move") {
        if (fid.empty() && path.empty())
        {
            fid = parameter;
            appliedRules.NoSuchFolderAction = "fallback_to_inbox";
        }
    } else if (action.Type == "delete") {
        if (fid.empty() && path.empty())
        {
            path = "\\trash";
            appliedRules.NoSuchFolderAction = "fail";
        }
    } else if (action.Type == "movel") {
        appliedRules.Lids.push_back(parameter);
    } else if (action.Type == "status") {
        if (parameter == "RO") {
            appliedRules.LabelSymbols.emplace_back("seen_label");
        } else if (parameter == "OQ") {
            appliedRules.LabelSymbols.emplace_back("answered_label");
        } else if (parameter == "PL") {
            appliedRules.LabelSymbols.emplace_back("forwarded_label");
        }
    } else if (action.Type == "forward") {
        appliedRules.Forwards.push_back(parameter);
        applyResult = EApplyResult::WithoutStore;
    } else if (action.Type == "forwardwithstore") {
        appliedRules.Forwards.push_back(parameter);
    } else if (action.Type == "reply") {
        appliedRules.Replies.emplace_back(TAutoreply{MakeAutoreplyAddress(message), parameter});
    } else if (action.Type == "notify") {
        appliedRules.Notifies.push_back(parameter);
    } else {
        return {};
    }

    return applyResult;
}

TApplyResult ProcessApplyActionResult(const std::optional<EApplyResult>& applyResult)
{
    TApplyResult actionApplyResult;
    if (applyResult) {
        if (*applyResult == EApplyResult::WithStore) {
            actionApplyResult.appliedWithStore = true;
        } else if (*applyResult == EApplyResult::WithoutStore) {
            actionApplyResult.appliedWithoutStore = true;
        }
    }

    return actionApplyResult;
}

std::string MakeAutoreplyAddress(const TMessage& message) {
    if (message.ReplyTo) {
        return *message.ReplyTo;
    }

    if (message.From) {
        return message.From->Local + "@" + message.From->Domain;
    }

    if (message.Sender) {
        return *message.Sender;
    }

    return {};
}

}
