#include "server.h"
#include "session.h"
#include "http/handlers/ping.h"
#include "http/handlers/store.h"
#include <mail/notsolitesrv/src/config/recognizer.h>
#include <mail/notsolitesrv/src/firstline/firstline.h>
#include <mail/notsolitesrv/src/firstline/firstline_impl.h>
#include <mail/notsolitesrv/src/furita/client.h>
#include <mail/notsolitesrv/src/mdbsave/client.h>
#include <mail/notsolitesrv/src/meta_save_op/meta_save_op.h>
#include <mail/notsolitesrv/src/msearch/client_impl.h>
#include <mail/notsolitesrv/src/msettings/client_impl.h>
#include <mail/notsolitesrv/src/mthr/mthr.h>
#include <mail/notsolitesrv/src/rules/domain/domain_rules.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/tupita/client.h>
#include <mail/notsolitesrv/src/util/find_module.h>
#include <mail/notsolitesrv/src/util/recognizer.h>
#include <yplatform/log.h>
#include <yplatform/ptree.h>
#include <yplatform/reactor.h>
#include <yplatform/find.h>
#include <ymod_smtpserver/server.h>
#include <ymod_webserver/server.h>
#include <util/generic/is_in.h>
#include <shared_mutex>

namespace NNotSoLiteSrv {

namespace {

NFurita::TFuritaClientPtr MakeFuritaClient(TContextPtr ctx) {
    const auto& config = *ctx->GetConfig();
    auto clusterCall = NUtil::FindClusterCallModule(config.Furita->ClusterClient);
    auto tvmModule = NUtil::FindTvmModule(config.Furita->TvmModule);
    return std::make_shared<NFurita::TFuritaClient>(std::move(ctx), std::move(clusterCall),
        std::move(tvmModule));
}

NMdbSave::TMdbSaveClientPtr MakeMdbSaveClient(TContextPtr ctx) {
    const auto& config = *ctx->GetConfig();
    auto clusterCall = NUtil::FindClusterCallModule(config.MdbSave->ClusterClient);
    auto tvmModule = NUtil::FindTvmModule(config.MdbSave->TvmModule);
    return std::make_shared<NMdbSave::TMdbSaveClient>(std::move(ctx), std::move(clusterCall),
        std::move(tvmModule));
}

NMSearch::TMSearchClientPtr MakeMSearchClient(TContextPtr ctx) {
    const auto& config = *ctx->GetConfig();
    auto clusterClient = NUtil::FindClusterCallModule(config.MSearch->ClusterClient);
    auto tvmModule = NUtil::FindTvmModule(config.MSearch->TvmModule);

    return std::make_shared<NMSearch::TMSearchClient>(ctx, std::move(clusterClient), std::move(tvmModule));
}

NMSettings::TMSettingsClientPtr MakeMSettingsClient(boost::asio::io_context& io, TConfigPtr config) {
    auto clusterClient = NUtil::FindClusterCallModule(config->MSettings->ClusterClient);
    auto tvmModule = NUtil::FindTvmModule(config->MSettings->TvmModule);

    return std::make_shared<NMSettings::TMSettingsClient>(io, std::move(config), std::move(clusterClient), std::move(tvmModule));
}

NTupita::TTupitaClientPtr MakeTupitaClient(TContextPtr ctx) {
    auto clusterCall = NUtil::FindClusterCallModule(ctx->GetConfig()->Tupita->ClusterClientName);
    return std::make_shared<NTupita::TTupitaClient>(std::move(ctx), std::move(clusterCall));
}

NRules::TDomainRulesClients MakeDomainRulesClients(TContextPtr ctx) {
    return {MakeFuritaClient(ctx), MakeTupitaClient(ctx)};
}

NRules::TDomainRulesPtr MakeDomainRules(TContextPtr ctx, NUser::TStoragePtr userStorage,
    boost::asio::io_context& ioContext)
{
    auto clients{MakeDomainRulesClients(ctx)};
    return std::make_shared<NRules::TDomainRules>(std::move(clients), std::move(ctx), std::move(userStorage),
        ioContext);
}

NMetaSaveOp::TMetaSaveOpClients MakeMetaSaveOpClients(TContextPtr ctx) {
    return {MakeFuritaClient(ctx), MakeMdbSaveClient(ctx), MakeTupitaClient(ctx)};
}

NMetaSaveOp::TMetaSaveOpComponents MakeMetaSaveOpComponents(NFirstline::TFirstlinePtr firstline, NMthr::TMthrPtr mthr) {
    return {std::move(firstline), std::move(mthr)};
}

NMetaSaveOp::TMetaSaveOpPtr MakeMetaSaveOp(TContextPtr ctx, NFirstline::TFirstlinePtr firstline,
    NMthr::TMthrPtr mthr, NUser::TStoragePtr userStorage, boost::asio::io_context& ioContext)
{
    auto clients = MakeMetaSaveOpClients(ctx);
    auto components = MakeMetaSaveOpComponents(std::move(firstline), std::move(mthr));
    return std::make_shared<NMetaSaveOp::TMetaSaveOp>(std::move(clients), std::move(components),
        std::move(ctx), std::move(userStorage), ioContext);
}

}

class TServer: public IServer {
private:
    using TMutex = std::shared_timed_mutex;
    using TReadLock = std::shared_lock<TMutex>;
    using TWriteLock = std::unique_lock<TMutex>;

public:
    void init(const yplatform::ptree& conf) override {
        ReadConfig(conf);
        InitRecognizer();

        Firstline = std::make_shared<NFirstline::TFirstline>(
            std::make_shared<NFirstline::NImpl::TFirstline>(GetConfig()->Firstline),
            *yplatform::find_reactor(GetConfig()->Firstline->Reactor)->io()
        );
        Mthr = std::make_shared<NMthr::TMthr>(GetConfig());

        auto smtpServer = yplatform::find<ymod_smtpserver::Server>("smtp_server");
        smtpServer->setFactory(std::make_shared<TSessionFactory>());

        Reactor = yplatform::find_reactor(conf.get("reactor", "global"));
        BindHttpHandlers();
    }

    void reload(const yplatform::ptree& conf) override {
        ReadConfig(conf);
        // Don't support reloading recognize settings because operation isn't thread safe
    }

    void Deliver(
        TContextPtr ctx,
        std::shared_ptr<std::string> msg,
        const TEnvelope& envelope,
        NUser::TStoragePtr userStorage,
        const NTimeTraits::TSystemTimePoint& deliveryStartMark,
        NNewEmails::TProcessor::TAsyncSender newEmailsSender,
        TDeliverer::TCallback cb) override
    {
        Reactor->io()->post(
            [
                ctx,
                reactor = Reactor,
                firstline = Firstline,
                mthr = Mthr,
                msg,
                envelope,
                userStorage,
                deliveryStartMark,
                newEmailsSender = std::move(newEmailsSender),
                cb = std::move(cb)
            ]() {
                auto& ioContext = *reactor->io();
                auto dlv = std::make_shared<TDeliverer>(
                    &ioContext,
                    ctx,
                    MakeDomainRules(ctx, userStorage, ioContext),
                    MakeMetaSaveOp(ctx, firstline, mthr, userStorage, ioContext),
                    msg,
                    envelope,
                    userStorage,
                    deliveryStartMark,
                    newEmailsSender,
                    cb);
                dlv->SetMSearchClient(MakeMSearchClient(ctx));
                dlv->SetMSettingsClient(MakeMSettingsClient(ioContext, ctx->GetConfig()));
                yplatform::spawn(dlv);
            });
    }

    void Deliver(
        TContextPtr ctx,
        const std::string& stid,
        const TEnvelope& envelope,
        NUser::TStoragePtr userStorage,
        const NTimeTraits::TSystemTimePoint& deliveryStartMark,
        TDeliverer::TMessageProcessor syncMessageProcessor,
        NNewEmails::TProcessor::TAsyncSender newEmailsSender,
        TDeliverer::TCallback cb) override
    {
        Reactor->io()->post(
            [
                ctx,
                reactor = Reactor,
                firstline = Firstline,
                mthr = Mthr,
                stid,
                envelope,
                userStorage,
                deliveryStartMark,
                syncMessageProcessor = std::move(syncMessageProcessor),
                newEmailsSender,
                cb = std::move(cb)
            ]() {
                auto& ioContext = *reactor->io();
                auto dlv = std::make_shared<TDeliverer>(
                    &ioContext,
                    ctx,
                    MakeDomainRules(ctx, userStorage, ioContext),
                    MakeMetaSaveOp(ctx, firstline, mthr, userStorage, ioContext),
                    stid,
                    envelope,
                    userStorage,
                    deliveryStartMark,
                    syncMessageProcessor,
                    newEmailsSender,
                    cb);
                dlv->SetMSearchClient(MakeMSearchClient(ctx));
                dlv->SetMSettingsClient(MakeMSettingsClient(ioContext, ctx->GetConfig()));
                yplatform::spawn(dlv);
            });
    }

    TConfigPtr GetConfig() const override {
        TReadLock lock(Mutex);
        return Config;
    }

    NTskv::TUserJournalPtr GetUserJournalWriter() const override {
        TReadLock lock(Mutex);
        return UJWriter;
    }

private:
    void ReadConfig(const yplatform::ptree& conf) {
        try {
            auto newConfig = std::make_shared<TConfig>(conf);
            if (newConfig) {
                TWriteLock lock(Mutex);
                Config = newConfig;
                UJWriter = NTskv::CreateUserJournalWriter(Config);
            }
        } catch (const std::exception& e) {
            LOGDOG_(
                MakeNslsLogger(),
                error,
                logdog::message="Failed to read config",
                logdog::exception=e);
        }
    }

    void BindHttpHandlers() const {
        auto webserver = yplatform::find<ymod_webserver::server>("web_server");
        BindHttpHanlder(webserver, std::make_shared<NHttp::TPing>());
        BindHttpHanlder(webserver, std::make_shared<NHttp::TStore>());
    }

    template <typename TWebServerPtr>
    void BindHttpHanlder(TWebServerPtr webserver, NHttp::THandlerPtr handler) const {
        auto callback = [this, self=shared_from_this(), handler](NHttp::THttpStreamPtr stream) {
            if (!IsIn(handler->AllowedMethods(), stream->request()->method)) {
                LOGDOG_(
                    MakeNslsLogger(),
                    error,
                    log::connection_id=stream->ctx()->uniq_id(),
                    logdog::message=
                        std::to_string(stream->request()->method) + " is not allowed for " +
                        handler->Name() + " request");
                return stream->result(NHttp::ECode::method_not_allowed);
            }

            try {
                auto ctx = std::make_shared<TContext>(GetConfig(), stream->ctx()->uniq_id());
                handler->Run(ctx, stream);
            } catch (const std::exception& e) {
                LOGDOG_(
                    MakeNslsLogger(),
                    error,
                    log::uniq_id=stream->ctx()->uniq_id(),
                    logdog::message="Error",
                    logdog::exception=e);
            } catch (...) {
                LOGDOG_(
                    MakeNslsLogger(),
                    error,
                    log::uniq_id=stream->ctx()->uniq_id(),
                    logdog::message="Unknown error while process HTTP request");
            }
        };
        webserver->bind("", handler->Paths(), std::move(callback));
    }

    void InitRecognizer() {
        if (!Config) {
            throw std::runtime_error("config is not valid");
        }

        NNotSoLiteSrv::NUtil::InitRecognizer(
            Config->Recognizer->LanguageDict,
            Config->Recognizer->LanguageWeights,
            Config->Recognizer->EncodingDict
        );
    }

private:
    yplatform::reactor_ptr Reactor;
    TConfigPtr Config;
    NFirstline::TFirstlinePtr Firstline;
    NMthr::TMthrPtr Mthr;
    NTskv::TUserJournalPtr UJWriter;
    mutable TMutex Mutex;
};

} // namespace NNotSoLiteSrv

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(NNotSoLiteSrv::TServer)
