#include "session.h"
#include "context.h"
#include "envelope.h"
#include "errors.h"
#include "server.h"
#include <mail/notsolitesrv/src/config/session.h>
#include <mail/notsolitesrv/src/smtp/sender.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/user/storage.h>
#include <yadns/ares_resolver_service.h>
#include <yadns/basic_dns_resolver.h>
#include <yplatform/find.h>
#include <yplatform/future/future.hpp>
#include <boost/asio/ip/host_name.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/variant/static_visitor.hpp>
#include <util/generic/algorithm.h>
#include <util/string/ascii.h>
#include <util/string/hex.h>

namespace NNotSoLiteSrv {

inline std::string ToString(const ymod_smtpserver::commands::Params& params) {
    std::string ret;
    for (const auto& [key, value]: params) {
        if (!ret.empty()) {
            ret.append(", ");
        }
        ret.append(key).append("=").append(value);
    }
    return ret;
}

inline char ByteToHex(uint8_t b) {
    return AsciiToLower(DigitToChar(b));
}

std::pair<std::string, std::string> GetRemoteAddressString(boost::asio::ip::address addr) {
    if (addr.is_v6()) {
        auto ipv6 = addr.to_v6();
        if (ipv6.is_v4_mapped() || ipv6.is_v4_compatible()) {
            addr = ipv6.to_v4();
        }
    }

    std::string lookupString;
    if (addr.is_v4()) {
        auto bytes = addr.to_v4().to_bytes();
        for (auto it = bytes.rbegin(); it != bytes.rend(); ++it) {
            lookupString.append(std::to_string(*it)).append(".");
        }
        lookupString.append("in-addr.arpa.");
    } else if (addr.is_v6()) {
        std::ostringstream os;
        auto bytes = addr.to_v6().to_bytes();
        for (auto it = bytes.rbegin(); it != bytes.rend(); ++it) {
            os << ByteToHex(*it % 16) << "." << ByteToHex(*it / 16) << ".";
        }
        lookupString = os.str() + "ip6.arpa.";
    }

    return std::make_pair(addr.to_string(), lookupString);
}

namespace NDetail {

using TResolver = basic_dns_resolver<ares_resolver_service>;

class TSession
    : public ymod_smtpserver::Session
    , public std::enable_shared_from_this<TSession>
    , private boost::static_visitor<>
{
public:
    using TConnectionPtr = ymod_smtpserver::ConnectionPtr;

    enum class EState {
        Start,
        Lhlo,
        MailFrom,
        RcptTo,
        Data,
        DataEnd,
        Quit
    };

    TSession(TConfigPtr cfg, TConnectionPtr conn)
        : ymod_smtpserver::Session(conn)
        , Ctx(std::make_shared<TContext>(cfg))
        , Connection(conn)
        , UserStoragePtr(std::make_shared<NUser::TStorage>())
        , LmtpServer(yplatform::find<NNotSoLiteSrv::IServer, std::shared_ptr>("nsls"))
    {}

    void start() final {
        namespace ph = std::placeholders;

        auto [remoteIp, lookupString] = GetRemoteAddressString(Connection->remoteAddr());
        const auto& ip = remoteIp;
        NSLS_LOG_CTX_DEBUG(logdog::message="start session, connect from [" + ip + "]");
        Envelope.RemoteIp = ip;
        Envelope.RemoteHost = ip;
        if (Ctx->GetConfig()->Session->ResolveIp) {
            TResolver resolver(Connection->ioService());
            resolver.async_resolve_ptr(
                lookupString,
                std::bind(&TSession::OnRemoteIpResolved, shared_from_this(), ph::_1, ph::_2));
        } else {
            Greeting();
        }
    }

    bool CheckCmdAfterQuit(const std::string& cmdName = "") {
        if (State == EState::Quit) {
            NSLS_LOG_CTX_ERROR(logdog::message=cmdName + " command after QUIT, something goes wrong");
            Connection->close();
            return true;
        }

        return false;
    }

    template <typename UnsupportedCmd>
    void operator()(UnsupportedCmd) {
        NSLS_LOG_CTX_ERROR(logdog::message="unsupported command");
        if (CheckCmdAfterQuit("UNSUPPORTED")) {
            return;
        }
        WriteRead(TResponse(502, TEnhancedCode(551)), "unsupported command");
    }

    void operator()(const ymod_smtpserver::commands::Unknown& unknown) {
        NSLS_LOG_CTX_ERROR(logdog::message="unknown command: '" + boost::trim_copy(unknown.ctx) + "'");
        if (CheckCmdAfterQuit("UNKNOWN")) {
            return;
        }
        WriteRead(TResponse(500, TEnhancedCode(551)), "unknown");
    }

    void operator()(const ymod_smtpserver::commands::SyntaxError& syntaxError) {
        NSLS_LOG_CTX_ERROR(logdog::message="syntax error = '" + boost::trim_copy(syntaxError.ctx) + "'");
        if (CheckCmdAfterQuit()) {
            return;
        }
        WriteRead(TResponse(555, TEnhancedCode(552)), "syntax error");
    }

    void operator()(const ymod_smtpserver::commands::Quit&) {
        NSLS_LOG_CTX_DEBUG(logdog::message="QUIT");
        if (CheckCmdAfterQuit("QUIT")) {
            return;
        }
        State = EState::Quit;
        Connection->writeResponse(
            TResponse(
                221,
                ymod_smtpserver::response_description(221) + "; " + Ctx->GetFullSessionId(),
                TEnhancedCode(200)),
            [this, self = shared_from_this()](TErrorCode ec, size_t) {
                if (ec) {
                    NSLS_LOG_CTX_ERROR(logdog::message="write response error on QUIT: " + ec.message());
                }
                return Connection->close();
            });
    }

    void operator()(const ymod_smtpserver::commands::Lhlo& lhlo) {
        NSLS_LOG_CTX_DEBUG(logdog::message="LHLO from " + lhlo.name);
        if (CheckCmdAfterQuit("LHLO")) {
            return;
        }
        ResetSession();
        Envelope.Lhlo = lhlo.name;
        State = EState::Lhlo;
        std::ostringstream respText;
        respText << boost::asio::ip::host_name() << "\n"
            << "8BITMIME\n"
            << "PIPELINING\n"
            << "ENHANCEDSTATUSCODES\n";
        WriteRead(TResponse(250, respText.str()), "LHLO");
    }

    void operator()(const ymod_smtpserver::commands::Rset&) {
        NSLS_LOG_CTX_DEBUG(logdog::message="RSET");
        if (CheckCmdAfterQuit("RSET")) {
            return;
        }
        State = State > EState::Lhlo ? EState::Lhlo : State;
        ResetSession();
        WriteRead(TResponse(250, "Flushed", TEnhancedCode(215)), "RSET");
    }

    void operator()(const ymod_smtpserver::commands::RcptTo& rcpt) {
        NSLS_LOG_CTX_DEBUG(
            logdog::message="RCPT TO: <" + rcpt.addr + "> {" + ToString(rcpt.params) + "}");
        if (CheckCmdAfterQuit("RCPT TO")) {
            return;
        }
        if (State != EState::MailFrom && State != EState::RcptTo) {
            return WriteRead(TResponse(503, TEnhancedCode(551)), "RCPT TO");
        }
        State = EState::RcptTo;

        auto addr = rcpt.addr;
        auto addedUserIter = UserStoragePtr->AddUser(addr, true, true);
        Rcpts.emplace_back(addr);
        if (Ctx->GetConfig()->Session->RegistrationPlusHack) {
            std::string login, domain;
            constexpr char REGPLUS[] = "registration+";
            constexpr auto REGPLUS_LENGTH = sizeof(REGPLUS) - 1;
            if (NUtil::SplitEmail(addr, login, domain) &&
                boost::istarts_with(login, REGPLUS) &&
                login.length() > REGPLUS_LENGTH)
            {
                addedUserIter->second.DeliveryResult.Forwards.emplace_back(login.substr(REGPLUS_LENGTH) + "@yandex-team.ru");
            }
        }
        WriteRead(TResponse(250, "Ok", TEnhancedCode(210)), "RCPT TO");
    }

    void operator()(const ymod_smtpserver::commands::MailFrom& mfrom) {
        NSLS_LOG_CTX_DEBUG(logdog::message="MAIL FROM: <" + mfrom.addr + "> {" + ToString(mfrom.params) + "}");
        if (CheckCmdAfterQuit("MAIL FROM")) {
            return;
        }
        if (State == EState::DataEnd) {
            ResetSession();
        } else if (State != EState::Lhlo) {
            return WriteRead(TResponse(503, TEnhancedCode(551)), "MAIL FROM");
        }
        State = EState::MailFrom;
        Envelope.MailFrom = mfrom.addr;
        auto envIdIter = mfrom.params.find("ENVID");
        if (envIdIter != mfrom.params.end()) {
            Ctx->SetEnvelopeId(envIdIter->second);
        }
        WriteRead(TResponse(250, "Ok", TEnhancedCode(210)), "MAIL FROM");
    }

    void operator()(const ymod_smtpserver::commands::Data&) {
        NSLS_LOG_CTX_DEBUG(logdog::message="DATA");
        if (CheckCmdAfterQuit("DATA")) {
            return;
        }
        if (State != EState::RcptTo) {
            return WriteRead(TResponse(503, TEnhancedCode(551)), "DATA");
        }
        State = EState::Data;
        DataMark = NTimeTraits::Now();
        Connection->writeResponse(
            TResponse(
                354,
                "Start mail input, end with <CRLF>.<CRLF>; " + Ctx->GetFullSessionId()),
            [this, self = shared_from_this()](TErrorCode ec, size_t) {
                if (ec) {
                    NSLS_LOG_CTX_ERROR(logdog::message="write response error on DATA: " + ec.message());
                    return Connection->close();
                }
                ReadMessage();
            });
    }

private:
    using TCommand = ymod_smtpserver::Command;
    using TResponse = ymod_smtpserver::Response;
    using TEnhancedCode = ymod_smtpserver::EnhancedStatusCode;

    void Greeting() {
        State = EState::Start;
        WriteRead(TResponse(220, boost::asio::ip::host_name() + " LMTP"));
    }

    void ResetSession() {
        if (State != EState::Start) {
            Ctx->SetEnvelopeId();
            UserStoragePtr = std::make_shared<NUser::TStorage>();
        }
        Rcpts.clear();
        Envelope.MailFrom.clear();
    }

    void LogInObsoletedFormat(
        const std::string& rcptAddr,
        const NUser::TUser& rcpt,
        const TMessagePtr& message,
        const std::string& attachmentsLogStr) const
    {
        NSLS_LOG_CTX_NOTICE(
            log::uid=rcpt.Uid,
            logdog::message=
                "message stored in db=pg backed for " + rcptAddr + " " +
                "(suid=" + rcpt.Suid + "), " +
                "mid=" + rcpt.DeliveryResult.Mid + ", " +
                "tid=" + rcpt.DeliveryResult.Tid + ", " +
                "imap_id=" + rcpt.DeliveryResult.ImapId + ", " +
                "fid=" + rcpt.DeliveryResult.Fid);

        const auto hint = message->GetXYHintByUid(rcpt.Uid);
        NSLS_LOG_CTX_DEBUG(
            log::uid=rcpt.Uid,
            logdog::message=
                "stored message for " + rcptAddr + " " +
                "(suid=" + rcpt.Suid + "): " +
                "mid=" + rcpt.DeliveryResult.Mid + ", " +
                "tid=" + rcpt.DeliveryResult.Tid + ", " +
                "imap_id=" + rcpt.DeliveryResult.ImapId + ", " +
                "fid=" + rcpt.DeliveryResult.Fid + ", " +
                "hdr_date=" + NTimeTraits::ToString(message->GetDate()) + ", " +
                "received_date=" + NTimeTraits::ToString(message->GetReceivedDate()) + ", " +
                "attachments=( " + attachmentsLogStr + "), " +
                "lids={" + boost::join(rcpt.DeliveryResult.Lids, ",") + "}, " +
                "method_id=" + hint.method_id + ", " +
                "ipfrom=" + hint.ipfrom + ", " +
                "session_key=" + hint.session_key);
    }

    std::string MakeResponseMsg (const TResponse& resp) const {
        std::stringstream msg;
        msg << "send DATA_END response: " << resp;
        return msg.str();
    }

    std::string BuildObsoletedAttachmentsLog(const TMessagePtr& message) const {
        std::ostringstream os;
        for (const auto& attach: message->GetAttachments()) {
            os << "[hid=" << attach.first << ", type=" << attach.second.Type << ", name="
                << attach.second.Name << ", size=" << attach.second.Size << "]; ";
        }
        return os.str();
    }

    std::string MakeErrorDescription(const TErrorCode& errorCode) const {
        return "; " + errorCode.message();
    }

    void FillResponseMessage(TResponse& response, const NUser::TUser& rcpt) const {
        if (response.text.empty()) {
            response.text = ymod_smtpserver::response_description(response.code);
        }
        response.text += "; " + Ctx->GetFullSessionId();

        if (response.code == 250) {
            std::string mid = rcpt.DeliveryResult.Mid.empty() ? "none" : rcpt.DeliveryResult.Mid;
            std::string imapid;

            if (rcpt.DeliveryResult.IsDuplicate || rcpt.DeliveryResult.ImapId.empty()) {
                imapid = "none";
            } else {
                imapid = rcpt.DeliveryResult.ImapId;
            }

            response.text += " " + mid + " " + imapid;
        }

        if (rcpt.DeliveryResult.ErrorCode) {
            response.text += MakeErrorDescription(rcpt.DeliveryResult.ErrorCode);
        }
    }

    std::vector<TResponse> BuildResponses(TMessagePtr message) const {
        auto attachmentsLogStr = BuildObsoletedAttachmentsLog(message);
        std::vector<TResponse> responses;
        for (const auto& rcptAddr: Rcpts) {
            auto userIt = FindIfPtr(UserStoragePtr->GetFilteredUsers(NUser::FromRcptTo),
                [&rcptAddr](const auto& u) { return boost::iequals(rcptAddr, u.first); });
            if (!userIt) {
                NSLS_LOG_CTX_ERROR(logdog::message="rcpt with email=<" + rcptAddr + "> not found in storage");
                TResponse resp(554, "Unknown user", TEnhancedCode{511});
                NSLS_LOG_CTX_DEBUG(logdog::message=MakeResponseMsg(resp));
                responses.emplace_back(std::move(resp));
                continue;
            }

            const auto& rcpt = (*userIt).second;

            TResponse resp(221, "");
            if (rcpt.Status == NUser::ELoadStatus::Unknown) {
                resp.code = 451;
                resp.enhancedCode = 450;
            } else if (rcpt.Status != NUser::ELoadStatus::Found) {
                resp.code = 554;
                resp.enhancedCode = 511;
                resp.text = "Unknown user";
            } else if (NError::IsOk(rcpt.DeliveryResult.ErrorCode)) {
                resp.code = 250;
                resp.enhancedCode = 200;
                if (!rcpt.DeliveryResult.ErrorCode) {
                    LogInObsoletedFormat(rcptAddr, rcpt, message, attachmentsLogStr);
                    auto copyToInboxUserIt = FindIfPtr(UserStoragePtr->GetFilteredUsers(NUser::ForDelivery),
                        [&rcptAddr](const auto& u) { return u.second.DeliveryParams.CopyToInbox && boost::iequals(rcptAddr, u.first); });
                    if (copyToInboxUserIt && NError::IsOk(copyToInboxUserIt->second.DeliveryResult.ErrorCode)) {
                        LogInObsoletedFormat(rcptAddr, copyToInboxUserIt->second, message, attachmentsLogStr);
                    }
                } else if (rcpt.DeliveryResult.ErrorCode == EError::DeliveryLoopDetected) {
                    NSLS_LOG_CTX_NOTICE(logdog::message="loop detected, for " + rcptAddr + " (" + rcpt.Uid + ")");
                }
            } else if (rcpt.DeliveryResult.ErrorCode == EError::MetaSaveOpPermanentError) {
                // mdb_commit returns perm_error only in case of non-existent folder
                resp.code = 554;
                resp.enhancedCode = 521;
                resp.text = "Permanent store error!";
            } else if (NError::IsPermError(rcpt.DeliveryResult.ErrorCode)) {
                resp.code = 551;
                resp.enhancedCode = 530;
            } else {
                resp.code = 451;
                resp.enhancedCode = 450;
            }

            FillResponseMessage(resp, rcpt);
            NSLS_LOG_CTX_DEBUG(logdog::message=MakeResponseMsg(resp));
            responses.emplace_back(std::move(resp));
        }
        return responses;
    }

    void OnRemoteIpResolved(TErrorCode ec, TResolver::iterator_ptr it) {
        if (!ec) {
            Envelope.RemoteHost = *it;
            if (Envelope.RemoteHost.back() == '.') {
                Envelope.RemoteHost.pop_back();
            }
        } else {
            NSLS_LOG_CTX_ERROR(logdog::message=
                "failed to back-resolve ip " + Envelope.RemoteIp + ": " +
                ec.message() + ", use IP instead");
            Envelope.RemoteHost = Envelope.RemoteIp;
        }
        Greeting();
    }

    void OnDelivery(TErrorCode errc, TMessagePtr message) {
        std::vector<TResponse> responses;

        if (errc) {
            NSLS_LOG_CTX_ERROR(logdog::message="internal onDelivery error: " + errc.message());
            boost::optional<TResponse> resp;
            if (NError::IsPermError(errc)) {
                resp.emplace(554, TResponse::EnhancedStatusCodeOpt(530));
            } else {
                resp.emplace(451, TResponse::EnhancedStatusCodeOpt(450));
            }
            resp->text = ymod_smtpserver::response_description(resp->code) + "; "
                + Ctx->GetFullSessionId() + MakeErrorDescription(errc);

            FillN(std::back_inserter(responses), Rcpts.size(), *resp);
            NSLS_LOG_CTX_ERROR(logdog::message=MakeResponseMsg(*resp));
        } else {
            responses = BuildResponses(message);
        }

        Connection->writeResponses(
            responses,
            [this, self = shared_from_this()](TErrorCode ec = TErrorCode(), size_t = 0) {
                if (ec) {
                    NSLS_LOG_CTX_ERROR(logdog::message="DATA_END writeResponse error: " + ec.message());
                    return Connection->close();
                }
                ReadCommand();
            });
    }

    void WriteRead(TResponse response, const std::string& cmdName = "") {
        if (response.enhancedCode) {
            response.text.append("; ").append(Ctx->GetFullSessionId());
        }
        Connection->writeResponse(std::move(response),
            [cmdName, this, self = shared_from_this()](TErrorCode ec, size_t) {
                if (ec) {
                    std::string logCmd;
                    if (!cmdName.empty()) {
                        logCmd = " on " + cmdName;
                    }
                    NSLS_LOG_CTX_ERROR(logdog::message="write response error" + logCmd + ": " + ec.message());
                    return connection->close();
                }
                ReadCommand();
            });
    }

    void ReadCommand() {
        Connection->readCommand([this, self = shared_from_this()](TErrorCode ec, TCommand cmd) {
            if (ec) {
                if (ec != boost::asio::error::eof) {
                    NSLS_LOG_CTX_ERROR(logdog::message="read command error: " + ec.message());
                }
                return connection->close();
            }
            return boost::apply_visitor(*this, cmd);
        });
    }

    void ReadMessage() {
        Connection->readMessage(
            [this, self = shared_from_this()](TErrorCode ec, std::shared_ptr<std::string> msg) {
                if (ec) {
                    NSLS_LOG_CTX_ERROR(logdog::message="read message error: " + ec.message());
                    return Connection->close();
                }
                State = EState::DataEnd;
                NSLS_LOG_CTX_NOTICE(logdog::message=
                    "from=<" + Envelope.MailFrom + ">, " +
                    "nrcpts=" + std::to_string(Rcpts.size()) + ", "
                    "ip=" + Connection->remoteAddr().to_string());
                NSLS_LOG_CTX_NOTICE(logdog::message="got message, size=" + std::to_string(msg->size()));
                LmtpServer->Deliver(Ctx, msg, Envelope, UserStoragePtr, DataMark, &NSmtp::AsyncSmtpSender,
                    [this, self = shared_from_this()](TErrorCode errc, TMessagePtr message = TMessagePtr()) {
                        Connection->ioService().post([this, self = shared_from_this(), errc, message]() {
                            OnDelivery(errc, message);
                        });
                    });
            });
    }

    TContextPtr Ctx;
    TConnectionPtr Connection;

    std::vector<std::string> Rcpts;
    EState State = EState::Quit;
    NTimeTraits::TSystemTimePoint DataMark = NTimeTraits::Now();
    TEnvelope Envelope;
    NUser::TStoragePtr UserStoragePtr;
    TServerPtr LmtpServer;
};

} // namespace NDetail

TSessionFactory::TSessionPtr TSessionFactory::create(TSessionFactory::TConnectionPtr connection) {
    auto server = yplatform::find<IServer>("nsls");
    return std::make_shared<NDetail::TSession>(server->GetConfig(), connection);
}

} // namespace NNotSoLiteSrv
