#include "client.h"

#include "file_spool.h"

#include <mail/notsolitesrv/src/tskv/log.h>
#include <mail/notsolitesrv/src/util/optional.h>

#include <ymod_smtpclient/call.h>
#include <yplatform/find.h>
#include <yplatform/coroutine.h>
#include <yplatform/log.h>
#include <util/generic/algorithm.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <iterator>

namespace NNotSoLiteSrv::NSmtp {
namespace NDetail {

#include <yplatform/yield.h>

using NUtil::ConvertOptional;

class TSender {
public:
    using TYieldCtx = yplatform::yield_context<TSender>;
    using TClientPtr = std::shared_ptr<ymod_smtpclient::Call>;

    TSender(
        TContextPtr ctx,
        const NConfig::TSmtpClient& config,
        TClientPtr client,
        const TMailFrom& from,
        const std::vector<TRecipient>& recipients,
        const std::string& message,
        TCallback callback
    )
        : Ctx(ctx)
        , Config(config)
        , Client(client)
        , From(from)
        , Recipients(recipients)
        , Message(message)
        , Callback(std::move(callback))
        , TotalAttempts(config.Attempts)
    {}

    void operator()(
        TYieldCtx yctx,
        TErrorCode ec = TErrorCode(),
        const ymod_smtpclient::Response& response = ymod_smtpclient::Response())
    {
        try {
            reenter (yctx) {
                NSLS_LOG_CTX_NOTICE(
                    logdog::message="start sending emails for " + std::to_string(Recipients.size()) +
                        " recipient(s) in chunks by " + std::to_string(Config.MaxRecipients),
                    logdog::where_name=Where);

                FirstRcpt = Recipients.begin();
                while (FirstRcpt != Recipients.end()) {
                    if (std::distance(FirstRcpt, Recipients.end()) <= Config.MaxRecipients) {
                        LastRcpt = Recipients.end();
                    } else {
                        LastRcpt = FirstRcpt + Config.MaxRecipients;
                    }

                    Request = MakeRequest();
                    for (; CurrentAttempt <= TotalAttempts; ++CurrentAttempt) {
                        NSLS_LOG_CTX_NOTICE(
                            logdog::message="from=" + Request.mailfrom.email +
                                ", nrcpts=" + std::to_string(Request.rcpts.size()) +
                                ", msg_size=" + std::to_string(Request.message->size()),
                            logdog::where_name=Where);

                        yield Client->asyncRun(
                                Ctx->GetTaskContext(Where),
                                Request,
                                MakeOptions(),
                                yctx);

                        if (ec) {
                            NSLS_LOG_CTX_ERROR(
                                logdog::message="smtp_client error: " + ec.message(),
                                logdog::where_name=Where);
                            ec = EError::SmtpTemporaryError;
                        } else if (!response.session) {
                            NSLS_LOG_CTX_ERROR(
                                logdog::message="Unknown error while communicating with remote server",
                                logdog::where_name=Where);
                            ec = EError::SmtpUnknownError;
                            Response = "Unknown error";
                        } else {
                            std::ostringstream responseStream;
                            responseStream << "code=" << response.session->replyCode;
                            if (auto enhancedStatusCode = response.session->enhancedStatusCode) {
                                responseStream << ", enhanced_code=" << *enhancedStatusCode;
                            }

                            responseStream << ", text='" << response.session->data << "'";
                            NSLS_LOG_CTX_NOTICE(
                                logdog::message="req finished, " + responseStream.str(),
                                logdog::where_name=Where);

                            using namespace ymod_smtpclient::reply;
                            Response = response.session->data;
                            auto status = smtp_code_to_status(response.session->replyCode);
                            switch (status) {
                                case Status::Accept:
                                    ec = EError::Ok;
                                    break;
                                case Status::Tempfail:
                                    ec = EError::SmtpTemporaryError;
                                    break;
                                case Status::Reject:
                                    ec = EError::SmtpPermanentError;
                                    break;
                                case Status::Unknown:
                                    ec = EError::SmtpUnknownError;
                                    break;
                            }
                        }

                        if (!ec || (ec != EError::SmtpTemporaryError && ec != EError::SmtpUnknownError)) {
                            break;
                        } else {
                            NSLS_LOG_CTX_DEBUG(
                                logdog::message=
                                    "failed to send message (attempt " + std::to_string(CurrentAttempt) + " of " +
                                    std::to_string(TotalAttempts) + "): (" + ec.message() + ") " + Response,
                                logdog::where_name=Where);
                        }
                    }

                    if (ec && Config.OnError == TOnSmtpError::FallbackToFile && !Recipients.empty()) {
                        TFileSpoolMeta meta;
                        meta.Host = Config.Remote.host;
                        meta.Port = Config.Remote.port;
                        meta.MailFrom.Email = From.email;
                        meta.MailFrom.Envid = ConvertOptional(From.envid);
                        std::transform(FirstRcpt, LastRcpt, std::back_inserter(meta.Recipients),
                            [](const auto& rcpt){return TRecipientMeta{.Email = rcpt.email};});
                        ec = SaveMailToSpool(Ctx, Config.FileSpoolDir, meta, Message);
                    }

                    FirstRcpt = LastRcpt;
                }
            }
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                logdog::message="got exception",
                logdog::exception=e,
                logdog::where_name=Where);
            ec = EError::DeliveryInternal;
            return Callback(ec, "");
        }

        if (yctx.is_complete()) {
            Callback(ec, Response);
        }
    }

private:
    ymod_smtpclient::Request MakeRequest() const {
        ymod_smtpclient::RequestBuilder builder;

        builder.address(Config.Remote);
        builder.mailfrom(From);
        builder.addRcpts(FirstRcpt, LastRcpt);
        builder.message(Message);

        return builder.release();
    }

    ymod_smtpclient::Options MakeOptions() const {
        ymod_smtpclient::Options opt;
        opt.allowRcptToErrors = false;
        opt.useSsl = Config.UseSSL;
        opt.timeouts.connectAttempt = opt.timeouts.connect = Config.ConnectTimeout;
        opt.timeouts.data = Config.DataTimeout;

        return opt;
    }

    TContextPtr Ctx;
    NConfig::TSmtpClient Config;
    TClientPtr Client;
    const TMailFrom From;
    std::vector<TRecipient> Recipients;
    const std::string Message;
    TCallback Callback;
    ymod_smtpclient::Request Request;
    std::string Response;

    int CurrentAttempt = 1;
    const int TotalAttempts;

    TRecipients::iterator FirstRcpt;
    TRecipients::iterator LastRcpt;
    const std::string Where {"SMTP"};
};

#include <yplatform/unyield.h>

} // namespace NDetail

namespace {

void LogRecipients(TContextPtr ctx, const TRecipients& recipients) {
    std::string message{"recipient count = " + std::to_string(recipients.size())};
    if (!recipients.empty()) {
        auto transformer = [](const auto& recipient){return "\"" + recipient.email + "\"";};
        const std::string separator{", "};
        auto emails = boost::algorithm::join(recipients |
            boost::adaptors::transformed(std::move(transformer)), separator);
        message += ", recipients: " + std::move(emails);
    }

    const std::string where{"SMTP"};
    NSLS_LOG_NOTICE(ctx, logdog::message=message, logdog::where_name=where);
}

}

void AsyncSendMessage(
    TContextPtr ctx,
    const NConfig::TSmtpClient& config,
    const TMailFrom& from,
    const TRecipients& recipients,
    const std::string& message,
    TCallback callback)
{
    AsyncSendMessageWithYmodSmtpClient(
        ctx,
        config,
        from,
        recipients,
        message,
        yplatform::find<ymod_smtpclient::Call, std::shared_ptr>("smtp_client"),
        callback);
}

void AsyncSendMessageWithYmodSmtpClient(
    TContextPtr ctx,
    const NConfig::TSmtpClient& config,
    const TMailFrom& from,
    const TRecipients& recipients,
    const std::string& message,
    std::shared_ptr<ymod_smtpclient::Call> smtpClient,
    TCallback callback)
{
    LogRecipients(ctx, recipients);
    if (recipients.empty()) {
        return callback(EError::DeliveryNoRecipients, "");
    }

    auto sender = std::make_shared<NDetail::TSender>(
        ctx,
        config,
        smtpClient,
        from,
        recipients,
        message,
        std::move(callback));
    yplatform::spawn(sender);
}

} // namespace NNotSoLiteSrv::NSmtp
