#include "file_spool.h"

#include <mail/notsolitesrv/src/smtp/types/reflection/file_spool.h>
#include <mail/notsolitesrv/src/tskv/log.h>

#include <util/folder/path.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/system/file.h>
#include <yamail/data/serialization/yajl.h>

namespace NNotSoLiteSrv::NSmtp {

void SaveEmlToSpool(const TFileHandle& fh, const std::string& message) {
    if (fh.Pwrite(message.data(), message.size(), 0) != static_cast<i32>(message.size())) {
        throw std::runtime_error("Can't write eml to file");
    }
}

void SaveMetaToSpool(const TFileHandle& fh, const TFileSpoolMeta& meta) {
    const auto json{yamail::data::serialization::toJson(meta).str()};
    if (fh.Pwrite(json.data(), json.size(), 0) != static_cast<i32>(json.size())) {
        throw std::runtime_error("Can't write eml to file");
    }
}

constexpr int MAX_TRIES = 5;

TErrorCode SaveMailToSpool(TContextPtr ctx, const std::string& dir, const TFileSpoolMeta& meta,
    const std::string& message)
{
    Y_ENSURE(!dir.empty());
    auto pathPrefix = TFsPath(dir) / TFsPath(ctx->GetFullSessionId());
    TString fnamePrefix = pathPrefix;
    TFileHandle emlHandle;
    TFileHandle metaHandle;
    try {
        for (int tryNum = 0; tryNum < MAX_TRIES && !emlHandle.IsOpen(); ++tryNum) {
            if (tryNum != 0) {
                fnamePrefix = TString(pathPrefix) + "." + ToString(tryNum);
            }

            auto tmpHandle = TFileHandle(TString(fnamePrefix + ".eml"), WrOnly | CreateNew);
            emlHandle.Swap(tmpHandle);
        }

        if (!emlHandle.IsOpen()) {
            throw std::runtime_error("Can't open file " + fnamePrefix + ".eml");
        }

        auto tmpHandle = TFileHandle(TString(fnamePrefix + ".json"), WrOnly | CreateNew);
        if (!tmpHandle.IsOpen()) {
            throw std::runtime_error("Can't open file " + fnamePrefix + ".json");
        }

        metaHandle.Swap(tmpHandle);
        if (metaHandle.Flock(LOCK_EX | LOCK_NB) < 0) {
            throw std::runtime_error("Can't lock meta file");
        }

        SaveEmlToSpool(emlHandle, message);
        SaveMetaToSpool(metaHandle, meta);
    } catch (const std::exception& e) {
        NSLS_LOG_ERROR(
            ctx,
            logdog::message="Error while save mail to file spool",
            logdog::exception=e);
        if (emlHandle.IsOpen()) {
            unlink((fnamePrefix + ".eml").c_str());
        }
        if (metaHandle.IsOpen()) {
            unlink((fnamePrefix + ".json").c_str());
        }
        return EError::DeliveryInternal;
    }

    return {};
}

} // namespace NNotSoLiteSrv::NSmtp
