#include "meta.h"
#include <mail/notsolitesrv/src/message/xyandexhint.h>
#include <mail/notsolitesrv/src/reflection/xyandexhint.h>
#include <mail/notsolitesrv/src/tskv/logger.h>

#include <mail/yplatform/include/yplatform/coroutine.h>
#include <mail/yplatform/include/yplatform/future/multi_future.hpp>

#include <library/cpp/digest/md5/md5.h>

namespace NNotSoLiteSrv::NSmtp {

namespace NDetail {

namespace NFuture = yplatform::future;

bool IsMetaNeededForUser(const NUser::TUser& user, const TMessagePtr& msg) {
    return NUser::Found(user) &&
        user.DeliveryResult.ErrorCode &&
        !NError::IsPermError(user.DeliveryResult.ErrorCode) &&
        !msg->GetStid(user.Uid).empty() &&
        !msg->GetXYHintByUid(user.Uid).skip_meta_msg &&
        !msg->GetXYHintByUid(user.Uid).imap &&
        !msg->GetXYHintByUid(user.Uid).external_imap_id &&
        !msg->GetXYHintByUid(user.Uid).copy_to_inbox;
}

#include <mail/yplatform/include/yplatform/yield.h>
class TMetaSender {
public:
    using TYieldCtx = yplatform::yield_context<TMetaSender>;

    TMetaSender(
        TContextPtr ctx,
        const TEnvelope& envelope,
        TMessagePtr message,
        const std::string& origEml,
        NUser::TStoragePtr userStorage,
        TMessageSender sender,
        TMetaCallback callback
    )
        : Ctx(ctx)
        , Envelope(envelope)
        , Message(message)
        , Eml(origEml)
        , UserStorage(userStorage)
        , AsyncSendMessage(sender)
        , Callback(std::move(callback))
        , Users(userStorage->GetFilteredUsers(std::bind(&IsMetaNeededForUser, std::placeholders::_1, message)))
    {}

    void operator()(
        TYieldCtx yctx,
        TErrorCode ec = TErrorCode(),
        const std::string& response = "")
    {
        try {
            reenter (yctx) {
                if (Users.empty()) {
                    NSLS_LOG_CTX_ERROR(
                        logdog::message="empty recipients for meta mail, skip it",
                        logdog::where_name=Where);
                    ec = EError::DeliveryNoRecipients;
                    yield break;
                }
                BodyMd5 = MD5::Calc(Message->GetBody());

                NSLS_LOG_CTX_DEBUG(
                    logdog::message="start sending meta to " + std::to_string(Users.size()) + " rcpts with body MD5=" + BodyMd5,
                    logdog::where_name=Where);
                Processes.reserve(Users.size());
                std::transform(Users.cbegin(), Users.cend(), std::back_inserter(Processes),
                    [](const auto& user) { return TProcess(user.first, user.second.Uid); });
                Future = NFuture::future_multi_and(Processes, [](const auto& p) { return p.Future; });

                yield {
                    Future.add_callback([this, yctx]() {
                        TErrorCode ec;
                        try {
                            Future.get();
                        } catch (const std::exception& e) {
                            NSLS_LOG_CTX_ERROR(
                                logdog::message="meta message submit error",
                                logdog::exception=e,
                                logdog::where_name=Where);
                            ec = EError::DeliveryInternal;
                        }
                        yctx(ec);
                    });

                    size_t curProcIdx = 0;
                    for (const auto& [email, user]: Users) {
                        AsyncSendMessage(
                            Ctx,
                            *Ctx->GetConfig()->Meta,
                            MakeFrom(),
                            {TRecipient{email}},
                            MakeData(user.Uid),
                            [this, yctx, curProcIdx](TErrorCode ec, const std::string& response) {
                                auto& proc = Processes[curProcIdx];
                                if (ec) {
                                    NSLS_LOG_CTX_ERROR(
                                        logdog::message="meta message submit error for email <" + proc.Email + ">: " + response,
                                        logdog::where_name=Where);
                                }
                                UpdateRecipientsStatus(Users | NUser::ByUid(proc.Uid), ec);
                                proc.Promise.set();
                            });

                        ++curProcIdx;
                    }
                }

                if (ec) {
                    yield break;
                }
            }
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                logdog::message="metamail submit exception",
                logdog::exception=e,
                logdog::where_name=Where);
            return Callback(EError::DeliveryInternal, "");
        }

        if (yctx.is_complete()) {
            Callback(ec, response);
        }
    }

private:
    TMailFrom MakeFrom() const {
        TMailFrom mailFrom(Envelope.MailFrom);
        if (Ctx->GetConfig()->Meta->SendEnvid) {
            mailFrom.envid = Ctx->GetFullSessionId();
        }

        return mailFrom;
    }

    void UpdateRecipientsStatus(
        NUser::TStorage::TFilteredUsers<NUser::TStorage::TContainer>&& users,
        TErrorCode ec)
    {
        for (auto& [email, user]: users) {
            user.DeliveryResult.ErrorCode = ec;
            user.DeliveryResult.MetaSend = true;
        }
    }

    std::string MakeData(const std::string& uid) const {
        std::string xyHint = MakeXYHint(uid);
        std::string eml;

        eml.reserve(xyHint.size() + Eml.size());
        eml.append(xyHint).append(Eml);
        return eml;
    }

    std::string MakeXYHint(const std::string& uid) const {
        TXYandexHint hint;
        hint.source_stid = Message->GetStid(uid);
        hint.final_headers_len = Message->GetFinalHeadersLength() + Message->GetPersonalHeadersLength(uid);
        hint.skip_loop_prevention = true;
        hint.body_md5 = BodyMd5;
        return "X-Yandex-Hint: " + NReflection::MakeXYHintHeaderValue(hint) + "\r\n";
    }

    TContextPtr Ctx;
    const TEnvelope& Envelope;
    TMessagePtr Message;
    const std::string& Eml;
    NUser::TStoragePtr UserStorage;
    TMessageSender AsyncSendMessage;
    TMetaCallback Callback;
    NUser::TStorage::TFilteredUsers<NUser::TStorage::TContainer> Users;
    const std::string Where {"META"};

    std::string BodyMd5;

    struct TProcess {
        TProcess(const std::string& email, const std::string& uid)
            : Email(email)
            , Uid(uid)
        {
            Future = Promise;
        }

        NFuture::promise<void> Promise;
        NFuture::future<void> Future;
        std::string Email;
        std::string Uid;
    };
    std::vector<TProcess> Processes;
    NFuture::future<void> Future;
};
#include <mail/yplatform/include/yplatform/unyield.h>

} // namespace NDetail

void AsyncSendMeta(
    TContextPtr ctx,
    const TEnvelope& envelope,
    TMessagePtr message,
    const std::string& origEml,
    NUser::TStoragePtr userStorage,
    TMetaCallback callback)
{
    return AsyncSendMetaWithSender(ctx, envelope, message, origEml, userStorage, AsyncSendMessage, callback);
}

void AsyncSendMetaWithSender(
    TContextPtr ctx,
    const TEnvelope& envelope,
    TMessagePtr message,
    const std::string& origEml,
    NUser::TStoragePtr userStorage,
    TMessageSender sender,
    TMetaCallback callback)
{
    auto metaSender = std::make_shared<NDetail::TMetaSender>(
        ctx,
        envelope,
        message,
        origEml,
        userStorage,
        sender,
        callback);
    yplatform::spawn(metaSender);
}

} // namespace NNotSoLiteSrv::NSmtp
