#include "sender.h"
#include "client.h"
#include <yplatform/coroutine.h>
#include <iterator>

namespace NNotSoLiteSrv::NSmtp {

namespace NDetail {

std::vector<ymod_smtpclient::NotifyMode> TranslateNotifyMode(NNewEmails::TNotifyMode mode) {
    if (!(mode.Failure || mode.Success || mode.Delay)) {
        return {ymod_smtpclient::NotifyMode::Never};
    }

    std::vector<ymod_smtpclient::NotifyMode> ret;
    if (mode.Success) {
        ret.emplace_back(ymod_smtpclient::NotifyMode::Success);
    }
    if (mode.Delay) {
        ret.emplace_back(ymod_smtpclient::NotifyMode::Delay);
    }
    if (mode.Failure) {
        ret.emplace_back(ymod_smtpclient::NotifyMode::Failure);
    }
    return ret;
}

#include <yplatform/yield.h>
class TNewEmailsSender {
public:
    using TYieldCtx = yplatform::yield_context<TNewEmailsSender>;
    TNewEmailsSender(TContextPtr ctx, const NNewEmails::TDataProvider& dataProvider, TSenderCallback callback)
        : Ctx(ctx)
        , DataProvider(dataProvider)
        , Callback(std::move(callback))
    {
        auto rcpts = DataProvider.GetRecipients();
        std::transform(rcpts.begin(), rcpts.end(), std::back_inserter(Recipients),
            [](const auto& rcpt) {
                TRecipient ret{rcpt.Email};
                ret.notifyModes = TranslateNotifyMode(rcpt.NotifyMode);
                return ret;
            });
    }

    void operator()(TYieldCtx yctx, TErrorCode errc = {}, const std::string& response = "") {
        Y_UNUSED(response);
        try {
            reenter (yctx) {
                NSLS_LOG_CTX_NOTICE(
                    logdog::message="start sending emails for " + std::to_string(Recipients.size()) +
                        " recipient(s)",
                    logdog::where_name=Where);

                yield AsyncSendMessage(
                    Ctx,
                    DataProvider.GetConfig(),
                    DataProvider.GetMailFrom(),
                    Recipients,
                    DataProvider.GetMessage(),
                    yctx);
            }

            if (yctx.is_complete()) {
                return Callback(errc);
            }
        } catch (const std::exception& e) {
            return Callback(EError::DeliveryInternal);
        }
    }

private:
    TContextPtr Ctx;
    const NNewEmails::TDataProvider& DataProvider;
    TSenderCallback Callback;

    TRecipients Recipients;
    const std::string Where{"NEW_EMAILS_SENDER"};
};
#include <yplatform/unyield.h>

} // namespace NDetail

void AsyncSmtpSender(TContextPtr ctx, const NNewEmails::TDataProvider& dataProvider, TSenderCallback callback) {
    auto sender = std::make_shared<NDetail::TNewEmailsSender>(ctx, dataProvider, std::move(callback));
    yplatform::spawn(sender);
}

} // namespace NNotSoLiteSrv::NSmtp
