#include "logger.h"
#include <mail/notsolitesrv/src/util/string.h>
#include <mail/notsolitesrv/src/server.h>
#include <mail/library/utf8/utf8.h>

#include <user_journal/target.h>
#include <user_journal/operation.h>
#include <user_journal/parameters/operation_parameters.h>
#include <user_journal/parameters/parameter_id.h>
#include <user_journal/parameters/message.h>
#include <yplatform/log/typed.h>
#include <yplatform/log.h>
#include <yplatform/find.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/predicate.hpp>

#include <util/generic/algorithm.h>

namespace NNotSoLiteSrv::NTskv {

void LogAttachments(TContextPtr ctx, TMessagePtr message, const NUser::TUser& user) {
    auto attachments = message->GetAttachments();

    // MPROTO-3818 - log text/calendar attachments to tskv
    for (const auto& part: *message) {
        if (boost::iequals(part.GetContentType(), "text") &&
            boost::iequals(part.GetContentSubtype(), "calendar") &&
            attachments.find(part.GetHid()) == attachments.end())
        {
            attachments.emplace(part.GetHid(), TAttachment{part});
        }
    }

    if (attachments.empty()) {
        // No attachments - no logs
        return;
    }

    auto logger = yplatform::log::tskv_logger(YGLOBAL_LOG_SERVICE, "attachments");
    namespace t = yplatform::log::typed;
    for (const auto& attach: attachments) {
        YLOG(logger, info)
            << t::make_attr("uid", user.Uid)
            << t::make_attr("suid", user.Suid)
            << t::make_attr("mid", user.DeliveryResult.Mid)
            << t::make_attr("fidType", user.DeliveryResult.FolderName) // yes, I known that it's brain-damaged
            << t::make_attr("fidSpecType", user.DeliveryResult.FolderTypeCode)
            << t::make_attr("hid", attach.first)
            << t::make_attr("fileType", attach.second.Type)
            << t::make_attr("name", attach.second.Name)
            << t::make_attr("size", attach.second.Size)
            << t::make_attr("stid", message->GetStid())
            << t::make_attr("unixtime", time(nullptr))
            << t::make_attr("sessionId", ctx->GetFullSessionId());
    }
}

void LogSpamReport(TContextPtr /* ctx */, TMessagePtr /* message */, const NUser::TUser& /* user */) {
    // TODO: Not implemented yet
}

void LogUserJournal(TContextPtr ctx, TUserJournalPtr ujWriter, TMessagePtr message, const NUser::TUser& user) {
    using namespace user_journal::parameters;

    std::string ujState;
    std::string sender, envFrom;
    for (const auto& hdr: message->GetHeaders()) {
        if (boost::iequals(hdr.first, "x-mailer") ||
            boost::iequals(hdr.first, "x-mailru-msgtype") ||
            boost::iequals(hdr.first, "list-id"))
        {
            if (!ujState.empty()) {
                ujState.append(",");
            }
            ujState
                .append(hdr.first)
                .append("=\"")
                .append(::NUtil::Utf8Backslash(hdr.second))
                .append("\"");
        } else if (boost::iequals(hdr.first, "sender")) {
            sender = hdr.second;
        } else if (boost::iequals(hdr.first, "envelope-from")) {
            envFrom = hdr.second;
        }
    }

    std::string fromEmail;
    if (message->GetFrom() && !message->GetFrom()->empty()) {
        // use first "From" address
        fromEmail = message->GetFrom()->front().second;
    } else {
        // if "From" is empty try "Sender" or "Envelope-From" for logging
        if (sender.empty()) {
            fromEmail = envFrom;
        } else {
            fromEmail = sender;
        }
    }
    fromEmail = NUtil::StripBadChars(::NUtil::Utf8Sanitized(fromEmail));

    std::string toEmails, ccEmails;
    std::string phone;
    auto hint = message->GetXYHintByUid(user.Uid);
    if (user.DeliveryResult.FolderType == "sent") {
        auto getEmail = [](const TAddress::element_type::value_type& addr) -> std::string {
            return addr.first + (addr.first.empty() ? "" : " ") + "<" + addr.second + ">";
        };

        if (message->GetTo()) {
            toEmails = boost::join(*message->GetTo() | boost::adaptors::transformed(getEmail), ",");
        }
        toEmails = NUtil::StripBadChars(::NUtil::Utf8Sanitized(toEmails));

        if (message->GetCc()) {
            ccEmails = boost::join(*message->GetCc() | boost::adaptors::transformed(getEmail), ",");
        }
        ccEmails = NUtil::StripBadChars(::NUtil::Utf8Sanitized(ccEmails));

        phone = NUtil::StripBadChars(::NUtil::Utf8Sanitized(hint.phone));
    }

    std::string sessId = hint.session_id;
    if (sessId.empty()) {
        sessId = ctx->GetFullSessionId();
    }
    sessId = NUtil::StripBadChars(::NUtil::Utf8Sanitized(sessId));

    std::string ipFrom = hint.ipfrom;
    if (ipFrom.empty()) {
        ipFrom = "127.0.0.1"; // FIXME
    }
    ipFrom = NUtil::StripBadChars(::NUtil::Utf8Sanitized(ipFrom));
    bool hidden = boost::iequals(user.DeliveryResult.FolderType, "sent") ||
        IsIn(user.DeliveryResult.LabelSymbols, "draft_label");

    auto journal = ujWriter->createJournal(
        user.Uid,
        id::module("fastsrv"), // for compatibility reasons
        id::ip(ipFrom),
        id::suid(user.Suid),
        id::connectionId(sessId));

    journal.write<ReceiveMessage>(
        id::hidden(hidden),
        id::mdb("pg"),
        id::affected(1),
        id::state(ujState),
        id::fid(user.DeliveryResult.Fid),
        id::mid(user.DeliveryResult.Mid),
        id::tid(user.DeliveryResult.Tid),
        id::stid(message->GetStid()),
        id::ftype(user.DeliveryResult.FolderTypeCode),
        id::emailFrom(fromEmail),
        id::labelSymbols(user.DeliveryResult.LabelSymbols),
        id::filterIds(user.DeliveryResult.FilterIds),
        id::subject(message->GetSubject()),
        id::msgId(message->GetMessageId()),
        id::emailTo(toEmails),
        id::emailCc(ccEmails),
        id::phone(phone)
    );
}

void LogAll(TContextPtr ctx, TUserJournalPtr ujWriter, TMessagePtr message, const NUser::TUser& user) {
    LogAttachments(ctx, message, user);
    LogSpamReport(ctx, message, user);
    LogUserJournal(ctx, ujWriter, message, user);
}

} // namespace NNotSoLiteSrv::NTskv
