#include "client.h"

#include <mail/notsolitesrv/src/http/util.h>
#include <mail/notsolitesrv/src/tskv/log.h>
#include <mail/notsolitesrv/src/tupita/types/reflection/request.h>
#include <mail/notsolitesrv/src/tupita/types/reflection/response.h>

#include <yamail/data/deserialization/yajl.h>
#include <yamail/data/serialization/yajl.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/asio.hpp>
#include <boost/range/adaptor/transformed.hpp>

namespace NNotSoLiteSrv::NTupita {

static const std::string TUPITA{"TUPITA"};

TTupitaClient::TTupitaClient(TContextPtr ctx, TClusterCallPtr clusterCall)
    : Ctx{std::move(ctx)}
    , ClusterCall{std::move(clusterCall)}
{
}

void TTupitaClient::Check(boost::asio::io_context& ioContext, TUid uid, std::string requestId,
    TTupitaCheckRequest checkRequest, TCheckCallback callback) const
{
    auto clusterCallCallback = [=, executor = ioContext.get_executor()](auto&& errorCode, auto&& response) {
        boost::asio::post(executor, [=]{ProcessCheckResponse(std::move(errorCode), std::move(response),
            std::move(callback));});
    };

    ClusterCall->async_run(Ctx->GetTaskContext(TUPITA), MakeCheckRequest(uid, std::move(requestId),
        checkRequest), std::move(clusterCallCallback));
}

THttpRequest TTupitaClient::MakeCheckRequest(TUid uid, std::string requestId,
    const TTupitaCheckRequest& checkRequest) const
{
    auto url = "/check?uid=" + std::to_string(uid) + "&reqid=" + std::move(requestId);
    return THttpRequest::POST(std::move(url), yamail::data::serialization::toJson(checkRequest));
}

void TTupitaClient::ProcessCheckResponse(TErrorCode errorCode, THttpResponse response,
    TCheckCallback callback) const
{
    if (errorCode || (response.status != 200)) {
        ProcessError(std::move(errorCode), std::move(response), std::move(callback));
    } else {
        auto checkResult = ParseCheckResponse(std::move(response.body));
        if (!checkResult) {
            errorCode = make_error_code(EError::TupitaResponseParseError);
        }

        callback(std::move(errorCode), std::move(checkResult));
    }
}

TCheckResult TTupitaClient::ParseCheckResponse(std::string responseBody) const
{
    try {
        return yamail::data::deserialization::fromJson<TTupitaCheckResponse>(responseBody);
    } catch (const std::exception& ex) {
        NSLS_LOG_CTX_ERROR(log::message = "failed to parse Tupita check response", log::exception = ex,
            log::where_name = TUPITA);
        return {};
    }
}

void TTupitaClient::ProcessError(TErrorCode errorCode, THttpResponse response,
    TCheckCallback callback) const
{
    if (errorCode) {
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = TUPITA);
    } else if ((response.status / 100) != 5) {
        errorCode = make_error_code(EError::HttpNonRetryableStatus);
        NSLS_LOG_CTX_ERROR(log::message = MakeHttpStatusMessage(std::move(response)),
            log::error_code = errorCode, log::where_name = TUPITA);
    } else {
        errorCode = make_error_code(EError::HttpRetriesExceeded);
        NSLS_LOG_CTX_ERROR(log::message = MakeHttpStatusMessage(std::move(response)),
            log::error_code = errorCode, log::where_name = TUPITA);
    }

    callback(std::move(errorCode), {});
}

}
