#include "processor.h"
#include <mail/notsolitesrv/src/blackbox/client.h>
#include <mail/notsolitesrv/src/blackwhitelist/client.h>
#include <mail/notsolitesrv/src/message/part.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/library/utf8/utf8.h>

#include <yplatform/coroutine.h>
#include <yplatform/log.h>

#include <boost/asio/error.hpp>

namespace NNotSoLiteSrv::NUser {
namespace NDetail {

namespace NBW = NBlackWhiteList;
#include <yplatform/yield.h>

class TProcessor {
public:
    TProcessor(
        TContextPtr ctx,
        NBlackbox::TUserGetter&& resolveUser,
        NBlackWhiteList::TListsLoader&& loadLists,
        const std::string& email,
        bool isMailish,
        bool skipLists,
        TUser& userInfo,
        TCallback callback
    )
        : Ctx(ctx)
        , ResolveUser(std::move(resolveUser))
        , LoadLists(std::move(loadLists))
        , Email(email)
        , IsMailish(isMailish)
        , SkipLists(skipLists)
        , UserInfo(userInfo)
        , Callback(std::move(callback))
    {}

    void operator()(
        yplatform::yield_context<TProcessor> yctx,
        TErrorCode ec = TErrorCode(),
        NBW::TListPtr bwlist = NBW::TListPtr())
    {
        reenter (yctx) {
            NSLS_LOG_CTX_DEBUG(
                logdog::message="start processing " + Email,
                logdog::where_name=Where);

            yield return ResolveUser(Ctx, Email, IsMailish, UserInfo, yctx);

            if (ec || UserInfo.Status != NUser::ELoadStatus::Found) {
                if (!ec) {
                    ec = EError::UserNotFound;
                }
                NSLS_LOG_CTX_ERROR(
                    logdog::message="failed to resolve user [" + Email + "]: " + ec.message(),
                    logdog::where_name=Where);
                Result = ec;
                yield break;
            }

            if (IsMailish || SkipLists) {
                NSLS_LOG_CTX_DEBUG(
                    logdog::message=
                        "skip b/w lists loading for user " + Email + (IsMailish ? " (mailish)" : " (explicitly)"),
                    logdog::where_name=Where);
                Result = ec;
                yield break;
            }

            yield return LoadLists(Ctx, UserInfo.Uid, yctx);

            Result = ec;
            if (ec) {
                NSLS_LOG_CTX_ERROR(
                    logdog::message="failed to load blackwhitelist for [" + Email + "]: " + ec.message(),
                    logdog::where_name=Where);
                yield break;
            }

            UserInfo.DeliveryParams.BWList = bwlist;
        }

        if (yctx.is_complete()) {
            if (Result) {
                UserInfo.DeliveryResult.ErrorCode = Result;
            }
            Callback(Result);
        }
    }

private:
    TContextPtr Ctx;
    NBlackbox::TUserGetter ResolveUser;
    NBlackWhiteList::TListsLoader LoadLists;
    std::string Email;
    bool IsMailish;
    bool SkipLists;
    TUser& UserInfo;
    TCallback Callback;
    TErrorCode Result = TErrorCode();
    const std::string Where {"UP"};
};

#include <yplatform/unyield.h>

} // namespace NDetail

void ProcessUser(
    TContextPtr ctx,
    NBlackbox::TUserGetter resolver,
    NBlackWhiteList::TListsLoader loader,
    const std::string& email,
    bool isMailish,
    bool skipLists,
    TUser& userInfo,
    TCallback callback)
{
    auto processor = std::make_shared<NDetail::TProcessor>(
        ctx,
        std::move(resolver),
        std::move(loader),
        email,
        isMailish,
        skipLists,
        userInfo,
        std::move(callback));
    yplatform::spawn(processor);
}

} // namespace NNotSoLiteSrv::NUser
