#include "storage.h"
#include <mail/notsolitesrv/src/util/string.h>
#include <boost/algorithm/string/predicate.hpp>
#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>

namespace NNotSoLiteSrv::NUser {

TStorage::TContainer::iterator TStorage::AddUser(const std::string& email, bool isFromRcpt, bool needDelivery) {
    Y_ENSURE(!email.empty());
    Y_ENSURE(!IsMailish() || (!isFromRcpt && !needDelivery),
        "Cannot add users that needs delivery to mailish storage");

    if (auto it = Users.find(email); it != Users.end()) {
        if (it->second.DeliveryParams.IsFromRcptTo != isFromRcpt ||
            it->second.DeliveryParams.NeedDelivery != needDelivery)
        {
            it->second.DeliveryParams.IsFromRcptTo = it->second.DeliveryParams.IsFromRcptTo || isFromRcpt;
            it->second.DeliveryParams.NeedDelivery = it->second.DeliveryParams.NeedDelivery || needDelivery;
        }
        return it;
    }

    TUser user;
    user.DeliveryParams.NeedDelivery = needDelivery;
    user.DeliveryParams.IsFromRcptTo = isFromRcpt;
    return Users.emplace(email, user);
}

void TStorage::CloneAsCopyToInboxUser(
    const std::string& email,
    const TUser& user,
    TDeliveryId deliveryId)
{
    Y_ENSURE(!IsMailish());
    Y_ENSURE(!email.empty());
    Y_ENSURE(!deliveryId.empty() && AllOf(deliveryId, [](const char c) { return std::isdigit(c); }));
    Y_ENSURE(user.DeliveryParams.NeedDelivery);

    TUser newUser = user;
    newUser.DeliveryParams.DeliveryId = deliveryId;
    newUser.DeliveryParams.IsFromRcptTo = false;
    newUser.DeliveryParams.CopyToInbox = true;

    Users.emplace(email, newUser);
}

boost::optional<const TUser&> TStorage::GetUserByEmail(const std::string& email) const {
    auto iter = Users.find(email);
    if (iter == end()) {
        return {};
    }

    return iter->second;
}

boost::optional<TUser&> TStorage::GetUserByEmail(const std::string& email) {
    auto iter = Users.find(email);
    if (iter == end()) {
        return {};
    }

    return iter->second;
}

bool TStorage::SetMailish(const std::string& mailishUid) {
    if (mailishUid.empty() || !AllOf(mailishUid, [](char c) { return std::isdigit(c); })) {
        return false;
    }

    if (RecipientsCount() != 1) {
        return false;
    }

    Mailish = true;

    for (auto& [email, user]: GetFilteredUsers(FromRcptTo)) {
        if (user.DeliveryParams.IsFromRcptTo) {
            user.Uid = mailishUid;
        }
    }

    return true;
}

size_t TStorage::RecipientsCount() const {
    return GetFilteredUsers(FromRcptTo).size();
}

} // namepace NNotSoLiteSrv::NUser
