#pragma once

#include "predicates.h"
#include "user.h"
#include <mail/notsolitesrv/src/util/string.h>
#include <boost/optional.hpp>
#include <util/generic/noncopyable.h>
#include <map>
#include <memory>

namespace NNotSoLiteSrv::NUser {

class TStorage: private TNonCopyable {
public:
    using TEmail = std::string;
    using TContainer = std::multimap<TEmail, TUser, NUtil::TCaseInsensitiveCmp>;
    TStorage() = default;
    virtual ~TStorage() = default;

    TContainer::iterator AddUser(const TEmail& email, bool isFromRcpt = true, bool needDelivery = true);
    void CloneAsCopyToInboxUser(
        const TEmail& email,
        const TUser& user,
        TDeliveryId deliveryId);
    boost::optional<TUser&> GetUserByEmail(const TEmail& email);
    boost::optional<const TUser&> GetUserByEmail(const TEmail& email) const;

    bool SetMailish(const std::string& mailishUid);
    bool IsMailish() const { return Mailish; }

    auto begin() { return Users.begin(); }
    auto begin() const { return Users.begin(); }
    auto end() { return Users.end(); }
    auto end() const { return Users.end(); }
    auto size() const { return Users.size(); }
    size_t RecipientsCount() const;

    auto& GetUsers() { return Users; }
    const auto& GetUsers() const { return Users; }

    template <typename TContainer>
    class TFilteredUsers {
    public:
        using TPredicate = TFilterPredicate;
        using TBaseIterator = typename TContainer::iterator;
        using TBaseConstIterator = typename TContainer::const_iterator;

        TFilteredUsers(TContainer& users, TPredicate predicate)
            : Users(users)
            , Predicate(predicate)
        {}

        template <typename TBaseIterator>
        class TIterator: public std::iterator<std::forward_iterator_tag, typename TBaseIterator::value_type> {
        public:
            using TBase = TBaseIterator;
            using TSelf = TIterator<TBase>;
            TIterator(TBase base, TBase end, TPredicate predicate)
                : Base(base)
                , End(end)
                , Predicate(predicate)
            {
                GetNextMatchedUser();
            }

            TSelf operator++() {
                ++Base;
                GetNextMatchedUser();
                return *this;
            }

            auto& operator->() {
                return Base.operator->();
            }

            auto& operator*() {
                return Base.operator*();
            }

            bool operator==(const TSelf& other) const {
                return Base == other.Base;
            }

            bool operator==(const TBase& other) const {
                return Base == other;
            }

            bool operator!=(const TSelf& other) const {
                return Base != other.Base;
            }

            bool operator!=(const TBase& other) const {
                return Base != other;
            }

        private:
            void GetNextMatchedUser() {
                while (Base != End && !Predicate(Base->second)) {
                    ++Base;
                }
            }

            TBase Base;
            TBase End;
            TPredicate Predicate;
        };

        auto cbegin() const { return TIterator<TBaseConstIterator>(Users.begin(), Users.end(), Predicate); }
        auto cend() const { return TIterator<TBaseConstIterator>(Users.end(), Users.end(), Predicate); }

        auto begin() const {
            if constexpr (std::is_const<TContainer>()) {
                return TIterator<TBaseConstIterator>(Users.begin(), Users.end(), Predicate);
            } else {
                return TIterator<TBaseIterator>(Users.begin(), Users.end(), Predicate);
            }
        }
        auto end() const {
            if constexpr (std::is_const<TContainer>()) {
                return TIterator<TBaseConstIterator>(Users.end(), Users.end(), Predicate);
            } else {
                return TIterator<TBaseIterator>(Users.end(), Users.end(), Predicate);
            }
        }
        auto size() const {
            return std::distance(cbegin(), cend());
        }
        bool empty() const {
            return cbegin() == cend();
        }

        auto operator|(TPredicate pred) {
            return TFilteredUsers<TContainer>(Users, [oldPred = Predicate, newPred = std::move(pred)](TUser u) { return oldPred(u) && newPred(u); });
        }

        auto operator|(TPredicate pred) const {
            return TFilteredUsers<const TContainer>(Users, [oldPred = Predicate, newPred = std::move(pred)](TUser u) { return oldPred(u) && newPred(u); });
        }

    private:
        TContainer& Users;
        TPredicate Predicate;
    };

    auto GetFilteredUsers(TFilterPredicate predicate = All) { return TFilteredUsers<TContainer>(Users, std::move(predicate)); }
    auto GetFilteredUsers(TFilterPredicate predicate = All) const { return TFilteredUsers<const TContainer>(Users, std::move(predicate)); }

private:
    bool Mailish = false;
    TContainer Users;
};

using TStoragePtr = std::shared_ptr<TStorage>;

} // namespace NNotSoLiteSrv::NUser
