#include "headers.h"
#include "md5.h"
#include "recognizer.h"

#include <mail/library/utf8/utf8.h>
#include <mail/library/received_header/received_header.h>

#include <mimeparser/build_msg.h>
#include <mimeparser/ccnv.h>
#include <mimeparser/rfc2047.h>
#include <mail_getter/UTFizer.h>

#include <boost/asio/ip/host_name.hpp>

#include <butil/datetime/date_utils.h>
#include <util/generic/yexception.h>

#include <sstream>
#include <iomanip>

namespace NNotSoLiteSrv::NUtil {

namespace {

std::string DecodeRfc2047Helper(
    const std::vector<mulca_mime::EncodedWord>& words,
    const std::string& fallbackEncoding)
{
    std::string res;

    for (const auto& word: words) {
        if (word.word.empty()) {
            continue;
        }
        std::string wordCopy = word.word;
        const std::string& charset = (word.charset.empty() ? fallbackEncoding : word.charset);
        try {
            UTFizer::process(GetRecognizer(), charset, wordCopy);
        } catch (...) {
            mulca_mime::convert(charset.c_str(), "utf-8").doit(word.word, wordCopy);
        }
        res.append(wordCopy);
    }

    try {
        return mulca_mime::decode_numbered_entities(res, true);
    } catch (...) {
        return res;
    }
}

} // namespace

std::string DecodeHeaderRfc2047(const std::string& str, const std::string& fallbackEncoding) {
    auto encodedWords = mulca_mime::decode_rfc2047(str);
    return ::NUtil::Utf8Sanitized(DecodeRfc2047Helper(encodedWords, fallbackEncoding));
}

std::string MakeRfc2822Date(time_t date) {
    return DateUtils::rfc2822time(date, "Europe/Moscow");
}

std::string MakeXYForwardValue(const std::string& value) {
    std::string toHash = "_" + value + "_";

    return Md5HexDigest(toHash);
}

std::string MakeMessageId() {
    return generate_message_id();
}

std::pair<std::string, std::string> MakeReceivedHeader(
    const std::string& sessionId,
    const TEnvelope& envelope,
    const std::string& recipient)
{
    Y_ENSURE(!envelope.Lhlo.empty());
    Y_ENSURE(!envelope.RemoteHost.empty());
    Y_ENSURE(!envelope.RemoteIp.empty());

    NReceived::TReceived received;
    received.Date = MakeRfc2822Date(time(nullptr));

    received.LocalHost = envelope.Hostname;
    if (received.LocalHost.empty()) {
        received.LocalHost = boost::asio::ip::host_name();
    }
    received.HeloHost = envelope.Lhlo;
    received.RemoteHost = envelope.RemoteHost;
    received.RemoteIp = envelope.RemoteIp;
    received.Recipient = recipient;
    received.SessionId = sessionId;
    received.Protocol = "LMTP";
    return NReceived::BuildReceivedPair(received);
}

const char* CodecToLetter(ECodec codec) {
    switch (codec) {
        case ECodec::Base64:
            return "B";
        case ECodec::QuotedPrintable:
            return "Q";
    }
}

std::function<std::string(const std::string&)> GetCodecEncoder(ECodec codec) {
    switch (codec) {
        case ECodec::Base64:
            return &encode_base64;
        case ECodec::QuotedPrintable:
            return &encode_qp;
    }
}

std::string EncodeRfc2047(const std::string& str, ECodec codec, const std::string& charset) {
    return std::string{"=?"}.append(charset).append("?").append(CodecToLetter(codec)).append("?")
        .append(GetCodecEncoder(codec)(str)).append("?=");
}

} // namespace NNotSoLiteSrv::NUtil
