#include "string.h"
#include <mail/library/utf8/utf8.h>

#include <boost/range/istream_range.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/trim.hpp>

namespace NNotSoLiteSrv::NUtil {

bool TCaseInsensitiveCmp::operator()(const std::string& lhs, const std::string& rhs) const {
    const auto& locale = ::NUtil::GetDefaultUtfLocale();
    return boost::ilexicographical_compare(lhs, rhs, locale);
}

std::string DeBackslash(const std::string& val) {
    std::string ret;
    ret.reserve(val.size());

    bool escaped = false;
    for (auto ch: val) {
        if (ch == '\\' && !escaped) {
            escaped = true;
        } else {
            ret.push_back(ch);
            escaped = false;
        }
    }

    return ret;
}

std::string StripBadChars(const std::string& val) {
    auto pos = val.find('\0');

    if (pos != std::string::npos) {
        return val.substr(0, pos);
    }

    return val;
}

namespace {

const std::string ALLOWED_EMAIL_CHARS("!#$%&'*+/=?^_`}{|~)(-.");

} // namespace

bool ExtractLogin(const std::string& email, std::string& login) {
    using TIterator = std::string::const_iterator;
    using TIteratorRange = boost::iterator_range<TIterator>;

    TIteratorRange name(email.begin(), email.end());

    name = boost::trim_copy(name);
    if (name.empty()) {
        return false;
    }

    TIteratorRange domain(name.end(), name.end());
    auto at = std::find(name.begin(), name.end(), '@');
    if (at != name.end()) {
        domain = boost::make_iterator_range(at + 1, name.end());
        name = boost::make_iterator_range(name.begin(), at);
    }

    if (name.size() > 1 && name.front() == '\"' && name.back() == '\"') {
        name.advance_begin(1).advance_end(-1);
    }

    auto plus = std::find(name.begin(), name.end(), '+');

    if (plus != name.end() && (plus != name.begin())) {
        name = boost::make_iterator_range(name.begin(), plus);
    }

    for (auto ch: name) {
        const auto& allowedCmp = [ch](char a) { return ch == a; };
        if (static_cast<unsigned char>(ch) <= 0x7f &&
            !isalnum(ch) &&
            std::none_of(ALLOWED_EMAIL_CHARS.begin(), ALLOWED_EMAIL_CHARS.end(), allowedCmp))
        {
            return false;
        }
    }

    std::string ret;
    std::transform(name.begin(), name.end(), std::back_inserter(ret), tolower);
    ret.append("@");
    std::transform(domain.begin(), domain.end(), std::back_inserter(ret), tolower);

    login.swap(ret);
    return true;
}

bool SplitEmail(const std::string& email, std::string& local, std::string& domain) {
    auto atPos = email.find('@');
    if (atPos == std::string::npos) {
        return false;
    }
    local = email.substr(0, atPos);
    domain = email.substr(atPos + 1);
    return true;
}

} // namespace NNotSoLiteSrv::NUtil
