#!/usr/bin/env python

import argparse
import os
import re
import smtplib
import logging
from base64 import b64decode, b64encode
from datetime import datetime

from library.python.filelock import FileLock
from mail.notsolitesrv.tools.backup_duplicate.logins import USER_MAP


LOG_DIR = "/var/log/backup_duplicate"
LOCK_PATH = "/var/run/backup_duplicate.lock"

POSTFIX_HOST = "127.0.0.1"
POSTFIX_PORT = 25

BACKUP_PATH = "/backup/yandex-team"

HINT_RE = re.compile("^x-yandex-hint:", re.I)

app_log = logging.getLogger("app")
dup_log = logging.getLogger("duplicate")


def _validate_date(date):
    return datetime.strptime(date, "%Y-%m-%d").strftime("%Y-%m-%d")


def iter_emls(login, date):
    login_date_backup_path = os.path.join(BACKUP_PATH, login, date)
    _, _, eml_names = os.walk(login_date_backup_path).next()
    for eml_name in eml_names:
        eml_path = os.path.join(login_date_backup_path, eml_name)
        yield eml_name, eml_path


def rebuild_hint(login, lines):
    lines = map(str.strip, lines)
    header, value = lines[0].split(": ", 1)
    value += "".join(lines[1:])
    value = b64decode(value)
    value = value.replace("email=" + get_email(login), "email=" + get_email(USER_MAP[login]), 1)
    return header + ": " + b64encode(value) + "\n"


def build_xyhint(**kwargs):
    value = b64encode("\n".join("%s=%s" % (k, v) for k, v in kwargs.iteritems()))
    return "X-Yandex-Hint: " + value + "\n"


def get_mtime(path):
    return int(os.path.getmtime(path))


def prepare_message(login, eml_path):
    fd = open(eml_path)
    lines = fd.readlines()
    if lines and lines[0].startswith("From "):  # This non header line is added by postfix
        lines.pop(0)

    res = []
    i = 0
    while i < len(lines):
        line = lines[i]
        if not HINT_RE.match(line):
            res.append(line)
            i += 1
            continue

        hint_lines = [line]
        i += 1
        while i < len(lines) and lines[i].startswith((" ", "\t")):
            hint_lines.append(lines[i])
            i += 1
        res.append(rebuild_hint(login, hint_lines))

    mtime = get_mtime(eml_path)  # Prevent diff when comparing messages written to backup before 00:00 but sent after
    return "X-Yandex-Internal: 1\n" + build_xyhint(received_date=mtime) + "".join(res)


def send_message(login, msg):
    smtp = smtplib.SMTP(POSTFIX_HOST, POSTFIX_PORT, timeout=30.0)
    smtp.sendmail("mail-restore@yandex-team.ru", get_email(USER_MAP[login]), msg)
    smtp.quit()


def duplicate(login, date):
    duplicated = find_duplicated(login, date)
    total = 0
    for eml_name, eml_path in iter_emls(login, date):
        if eml_name in duplicated:
            continue
        try:
            msg = prepare_message(login, eml_path)
            send_message(login, msg)
            dup_log.info(eml_name)
            app_log.info("duplicate %s success", eml_name)
        except Exception as e:
            app_log.info("duplicate %s error: %s", eml_name, e)
            raise
        total += 1
    return total


def get_email(login):
    return login + "@yandex-team.ru"


def find_duplicated(login, date):
    fd = open(_get_dup_log_path(login, date))
    return {line.strip() for line in fd.xreadlines()}


def _get_dup_log_path(login, date):
    return os.path.join(LOG_DIR, login + "_" + date + ".log")


def setup_logging(login, date):
    handler = logging.FileHandler(os.path.join(LOG_DIR, "duplicate.log"))
    handler.setFormatter(logging.Formatter("[%(asctime)s] ({}) %(message)s".format(login)))
    app_log.addHandler(handler)
    app_log.setLevel(logging.DEBUG)

    log_path = _get_dup_log_path(login, date)
    dup_log.addHandler(logging.FileHandler(log_path))
    dup_log.setLevel(logging.DEBUG)


def print_logins(_):
    login_robot_map = [login + ":" + USER_MAP[login] for login in USER_MAP.keys()]
    print " ".join(login_robot_map)


def main(args):
    login, date = args.login, args.date
    setup_logging(login, date)
    app_log.info("Start duplicating for date: %s", date)
    lock = FileLock(LOCK_PATH)
    if lock.acquire(blocking=False):
        try:
            total = duplicate(login, date)
            app_log.info("Finished, total: %d", total)
            print "Ok"
        except Exception as e:
            app_log.exception("Finished with exc: %s", e)
            raise
    else:
        app_log.error("Failed to acquire lock on {}".format(LOCK_PATH))


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers()
    subparser = subparsers.add_parser("print-logins", help="Print available logins and exit")
    subparser.set_defaults(func=print_logins)

    subparser = subparsers.add_parser("duplicate", help="Duplicate messages for specified login and date")
    subparser.add_argument("login", choices=USER_MAP.keys())
    subparser.add_argument("date", type=_validate_date, help="e.g. 2018-09-20")
    subparser.set_defaults(func=main)
    args = parser.parse_args()
    args.func(args)
