#!/usr/bin/env python
# coding: utf-8

import argparse
import difflib
import os
import pprint
import requests
import logging

from collections import defaultdict
from datetime import datetime
from urllib import urlencode
from urlparse import urljoin

from lxml import etree

from pymdb.types.selected import MailCoordinates, Mail

from mail.notsolitesrv.tools.compare_mailboxes.mailbox_fetchers \
    import MailishMessageBoxFetcher, NormalUserMessageBoxFetcher

LOG_DIR = "/var/log/compare"
BB_BASE_URLS = {"bigml": "http://blackbox-mail.yandex.net",
                "corp": "http://blackbox.yandex-team.ru/blackbox",
                "testing": "http://pass-test.yandex.ru/blackbox"}
SHARPEI_URLS = {"bigml": "http://sharpei-production.mail.yandex.net",
                "corp": "http://sharpei-intranet-production.mail.yandex.net",
                "testing": "http://sharpei-testing.mail.yandex.net"}
FETCHERS = {"normal": NormalUserMessageBoxFetcher(),
            "mailish": MailishMessageBoxFetcher()}

VOLATILE_KEYS = {
    "coords": set(MailCoordinates._volatile + [
        "st_id", "size", "found_tid",
        "seen", "revision"
    ]),
    "mail": set(Mail._volatile + ["doom_date"]),
    "mime": {"offset_begin", "offset_end"},
}

log = logging.getLogger("app")


def diff_dicts(a, b, n=2):
    if a == b:
        return []

    diff_lines = list(difflib.ndiff(
        pprint.pformat(a, width=30, depth=3).splitlines(),
        pprint.pformat(b, width=30, depth=3).splitlines()))

    indices = [i for i, line in enumerate(diff_lines)
               if line.startswith(("+", "-"))]
    return diff_lines[max(0, indices[0] - n): indices[-1] + n]


def _validate_date(date):
    return datetime.strptime(date, "%Y-%m-%d").strftime("%Y-%m-%d")


def _build_userinfo_url(login, bb_url):
    query_args = {
        "method": "userinfo",
        "login": login,
        "sid": "smtp",
        "userip": "localhost",
        "dbfields": "account_info.country.uid",
    }
    endpoint = "blackbox/?" + urlencode(query_args)
    return urljoin(bb_url, endpoint)


def get_uid(login, bb_url):
    # user can be set in the form @uid
    if login[0] == '@':
        return login[1:]

    url = _build_userinfo_url(login, bb_url)
    resp = requests.get(url)
    if resp.status_code != 200:
        raise RuntimeError("BlackBox error: {}".format(resp.status_code))

    xml = resp.text
    root = etree.fromstring(xml.encode("utf-8"))
    uids = root.xpath("/doc/uid")
    if not uids:
        raise RuntimeError("Blackbox respond with no uids")

    uid = uids[0].text
    if uid is None:
        raise RuntimeError("Blackbox respond with empty uid")
    return int(uid)


def get_counts_per_folder(folders, mails):
    counts = defaultdict(int)
    for mail in mails.itervalues():
        folder = folders[mail.coords.fid]
        counts[folder.name] += 1
    return counts


def as_dict(item):
    if hasattr(item, "as_dict"):
        return {k: as_dict(v) for k, v in item.as_dict().iteritems()}
    elif isinstance(item, list):
        return [as_dict(i) for i in item]
    return item


def prepare_compare(mail):
    if mail["attaches"]:
        mail["attaches"].sort(key=lambda x: x["hid"])

    mail["mime"].sort(key=lambda x: x["hid"])

    attrs = mail["coords"]["attributes"]
    if "mulca-shared" in attrs:
        attrs.remove("mulca-shared")
    if "append" in attrs:
        attrs.remove("append")
    attrs.sort()


def remove_volatiles(mail):
    for k in VOLATILE_KEYS["mail"]:
        mail.pop(k, None)
    for k in VOLATILE_KEYS["coords"]:
        mail["coords"].pop(k, None)
    for mime in mail["mime"]:
        for k in VOLATILE_KEYS["mime"]:
            mime.pop(k, None)


def as_cmpable(mail):
    remove_volatiles(mail)
    prepare_compare(mail)
    return mail


def are_mails_equal(user_mails, robot_mails):
    equal = True
    for k in user_mails.viewkeys() | robot_mails.viewkeys():
        if k not in user_mails:
            log.info("No mail for user with params %s", k)
            equal = False
            continue

        if k not in robot_mails:
            log.info("No mail for robot with params %s", k)
            equal = False
            continue

        user_mail = as_cmpable(as_dict(user_mails[k]))
        robot_mail = as_cmpable(as_dict(robot_mails[k]))
        if user_mail != robot_mail:
            equal = False
            log.info("Mails with params %s are not equal, diff:", k)
            for line in diff_dicts(user_mail, robot_mail):
                log.info(line)

    return equal


def are_counts_per_folder_equal(left, right):
    if left == right:
        return True

    folders = left.viewkeys() | right.viewkeys()
    padding = max([len(f) for f in folders] or [0])
    for folder in folders:
        dist = left.get(folder, 0) - right.get(folder, 0)
        if dist != 0:
            log.info("%s: %d - %d = %d", folder.ljust(padding), left[folder], right[folder], dist)
    return False


def compare(sharpei_url, user_login, robot_login, date, bb_url, fetcher):
    log.info("Compare for date %s", date)

    user_folders, user_mails = fetcher.find_folders_and_mails(sharpei_url, get_uid(user_login, bb_url), date)
    robot_folders, robot_mails = fetcher.find_folders_and_mails(sharpei_url, get_uid(robot_login, bb_url), date)

    log.info("user_folders_cnt={0}, user_mails_cnt={1}".format(len(user_folders.keys()), len(user_mails.keys())))
    log.info("robot_folders_cnt={0}, robot_mails_cnt={1}".format(len(robot_folders.keys()), len(robot_mails.keys())))

    equal = are_mails_equal(user_mails, robot_mails)

    user_counts = get_counts_per_folder(user_folders, user_mails)
    robot_counts = get_counts_per_folder(robot_folders, robot_mails)

    if not are_counts_per_folder_equal(user_counts, robot_counts):
        equal = False
    return equal


def main(args):
    login, robot = args.login, args.robot
    date = args.date
    sharpei_url = SHARPEI_URLS[args.env]
    bb_url = BB_BASE_URLS[args.env]
    fetcher = FETCHERS[args.user_type]
    setup_logging(login, robot)
    try:
        if compare(sharpei_url, login, robot, date, bb_url, fetcher):
            log.info("Ok, mailboxes are equal for date: %s", date)
            print "Ok"
        else:
            log.info("Fail, mailboxes are not equal for date: %s", date)
    except Exception as e:
        log.exception("Exit with error: %s", e)
        raise


def setup_logging(login, robot):
    handler = logging.FileHandler(os.path.join(LOG_DIR, "compare.log"))
    handler.setFormatter(logging.Formatter("[%(asctime)s] ({login}, {robot}) %(message)s".format(login=login, robot=robot)))
    log.addHandler(handler)
    log.setLevel(logging.DEBUG)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("login", help="login at yandex or @<uid>")
    parser.add_argument("robot", help="login at yandex or @<uid>")
    parser.add_argument("date", type=_validate_date, help="e.g. 2018-09-20")
    parser.add_argument("env", choices=SHARPEI_URLS.keys())
    parser.add_argument("user_type", choices=FETCHERS.keys())
    parser.set_defaults(func=main)

    args = parser.parse_args()
    args.func(args)
