from mail.pypg.pypg.common import readonly_repeatable_read_transaction
from pymdb.queries import Queries
from pymdb import types

from datetime import datetime, timedelta
from urlparse import urljoin

import pytz
import requests

LOCAL_TZ = pytz.timezone("Europe/Moscow")
MDB_USER = "mxback"


def get_conninfo(sharpei_url, uid):
    url = urljoin(sharpei_url, "/conninfo?uid={}&mode=master".format(uid))
    resp = requests.get(url)
    if resp.status_code != 200:
        raise RuntimeError("Sharpei error: {}".format(resp.status_code))
    return resp.json()["addrs"][0]


def get_passwd():
    for line in open("/etc/yamail/.pgpass", "rt"):
        _, usr, pwd = line.strip().rsplit(":", 2)
        if usr == MDB_USER:
            return pwd
    raise RuntimeError("Unable to find pwd for '{}' in pgpass".format(MDB_USER))


def get_connstring(sharpei_url, uid):
    conn_info = get_conninfo(sharpei_url, uid)
    pwd = get_passwd()
    return "host={host} port={port} dbname={dbname} user={user} password={pwd}".format(pwd=pwd, user=MDB_USER,
                                                                                       **conn_info)


class NormalUserMessageBoxFetcher(object):
    def get_key(self, mail):
        return "fid={0}, hdr_date={1}, hdr_msg_id={2}".format(mail.coords.fid, mail.headers.hdr_date,
                                                              mail.headers.hdr_message_id)

    def is_from_calendar(self, mail):
        rcpt = next(r for r in mail.recipients if r.type == "from")
        return rcpt.email == "info@calendar.yandex-team.ru"

    def find_folders_and_mails(self, sharpei_url, uid, date):
        date = LOCAL_TZ.localize(datetime.strptime(date, "%Y-%m-%d"))
        max_received_date = datetime.now(LOCAL_TZ) - timedelta(hours=1)

        maildb = get_connstring(sharpei_url, uid)
        with readonly_repeatable_read_transaction(maildb) as conn:
            q = Queries(conn, uid)
            folders = {f.fid: f for f in q.folders()}
            mails = {}
            for mail in q.mails(min_received_date=date):
                if mail.coords.received_date > max_received_date or \
                        folders[mail.coords.fid].subscribed_for_shared_folder or \
                        "append" in mail.coords.attributes or \
                        self.is_from_calendar(mail) or \
                        mail.coords.fid == 6:  # Drafts
                    continue
                k = self.get_key(mail)
                assert k not in mails, "Two mails with same params %s for %s" % (k, uid)
                mails[k] = mail
            return folders, mails


class MailishMessageBoxFetcher(object):
    @staticmethod
    def get_key(fid, external_imap_id):
        return "fid={0}, external_imap_id={1}".format(fid, external_imap_id)

    def find_folders_and_mails(self, sharpei_url, uid, date):
        date = LOCAL_TZ.localize(datetime.strptime(date, "%Y-%m-%d"))
        max_received_date = datetime.now(LOCAL_TZ) - timedelta(hours=1)

        maildb = get_connstring(sharpei_url, uid)
        with readonly_repeatable_read_transaction(maildb) as conn:
            q = Queries(conn, uid)
            folders = {f.fid: f for f in q.folders()}
            mails = {}
            for mail in q.mailish_messages():
                external_imap_id = mail.imap_id
                mail = q.message(mid=mail.mid)
                mail = self.to_mail(mail)
                if mail.coords.received_date < date or mail.coords.received_date > max_received_date:
                    continue
                k = self.get_key(mail.coords.fid, external_imap_id)
                assert k not in mails, "Two mails with same params %s for %s" % (k, uid)
                mails[k] = mail
            return folders, mails

    def list_or_none(self, ItemClass, data):
        if data is not None:
            return [ItemClass(**i) for i in data]
        return None

    def to_mail(self, md):
        return types.Mail(
            mid=md.pop('mid'),
            coords=types.MailCoordinates(**md),
            headers=types.MailHeaders(**md),
            lids=md.pop('lids'),
            recipients=self.list_or_none(types.MailRecipient, md.pop('recipients')),
            attaches=self.list_or_none(types.MailAttach, md.pop('attaches')),
            mime=self.list_or_none(types.MailMimePart, md.pop('mime')),
            pop3="",  # random value added here just to make Mail constructor happy
            **md
        )
