#!/usr/bin/env python

import argparse
import requests
import smtplib
import logging
import sys

from urlparse import urljoin

from base64 import b64encode

LOG_FILE = "./resend_mails.log"
logging.getLogger().setLevel(logging.DEBUG)
log = logging.getLogger(__name__)


MDS_URL = "http://storage.mail.yandex.net:10010"
HOST = "localhost"
PORT = 5252
IGNORED_KEYS = ["source_stid", "final_headers_len", "body_md5", "phone"]


def build_xyhint(strhint):
    value = b64encode("\n".join("%s=%s" % (k, v) for k, v in filter(lambda (k, v): k not in IGNORED_KEYS, [s.split("=", 1) for s in strhint.split(" ")])))
    return "X-Yandex-Hint: " + value + "\r\n"


def prepare_message(msg, hints):
    hints.append("skip_loop_prevention=1")
    hint = ""
    for strhint in hints:
        one_hint = build_xyhint(strhint)
        log.info("Add hint: %s", one_hint)
        hint += one_hint
    return hint + msg


def send_message(emails, mailfrom, msg, host, port):
    log.info("Send msg: host=%s, port=%s, sender=%s, email=%s, msg_size=%s",
             host, port, mailfrom, ",".join(emails), len(msg))
    smtp = smtplib.SMTP(host, port, timeout=30.0)
    #  smtp.set_debuglevel(True)
    res = smtp.sendmail(mailfrom, emails, msg)
    log.info("Send msg: %s", res)
    smtp.quit()


def get_message(stid):
    query = "gate/get/%s" % stid
    url = urljoin(MDS_URL, query)

    log.info("Get msg from mds: url=%s", url)

    resp = requests.get(url)
    if resp.status_code != 200:
        log.error("Failed to get email from mds, stid=%s", stid)
        raise RuntimeError("Failed to get stid")

    log.info("Get msg from mds: size=%d", len(resp.content))

    return resp.content


def setup_logging():
    handler = logging.FileHandler(LOG_FILE)
    handler.setFormatter(logging.Formatter("[%(asctime)s] %(message)s"))
    log.addHandler(handler)
    log.setLevel(logging.DEBUG)


def get_data():
    for line in sys.stdin.readlines():
        data = eval(line)
        if len(data["email"]) != len(data["mulcaid"]):
            log.error("emails size != stids size: %s", line)
            continue
        if all(stid == data["mulcaid"][0] for stid in data["mulcaid"]):
            yield (data["mulcaid"][0], data["mailfrom"], data["email"], data["hint"])
        else:
            for i in range(len(data["mulcaid"])):
                yield (data["mulcaid"][i], data["mailfrom"], [data["email"][i]], data["hint"])


def main(host, port):
    setup_logging()
    log.info("Start")

    for stid, mailfrom, rcpts, hints in get_data():
        try:
            log.info("Start processing stid=%s, rcpts=%s", stid, ",".join(rcpts))
            msg = get_message(stid)
            msg = prepare_message(msg, hints)
            send_message(rcpts, mailfrom, msg, host, port)
        except Exception as e:
            log.error("Got error: %s", e)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--host", type=str, default=HOST, required=False, help="SMTP host")
    parser.add_argument("--port", type=int, default=PORT, required=False, help="SMTP port")
    args = parser.parse_args()
    main(args.host, args.port)
