#!/usr/bin/env python

import argparse
import requests
import smtplib
import logging
import sys
import re

from base64 import b64encode
from urlparse import urljoin

LOG_FILE = "./resend_mails_sent.log"
logging.getLogger().setLevel(logging.DEBUG)
log = logging.getLogger(__name__)


BB_URL = "http://blackbox.yandex-team.ru/blackbox"
DOMAIN = "yandex-team.ru"
MDS_URL = "http://storage.mail.yandex.net:10010"
HOST = "localhost"
PORT = 5252
HINT_SPAM_RE = re.compile(r"^x-yandex-hint:", re.I)
TM_RE = re.compile(r"^x-yandex-timemark: ?(?P<ts>\d+)", re.I)
IGNORED_UID_MID = {
    "1120000000070786": {"169729410956608001", "169729410956608002", "169729410956608003"},
    "1120000000049913": {"169729410956731791"},
    "1120000000159928": {"169729410956527748"},
    "1120000000157864": {"169729410956528587"},
    "1120000000073245": {"169729410956656401"},
    "1120000000106556": {"169729410956550263", "169729410956550265", "169729410956550267", "169729410956550269"},
    "1120000000142996": {"169729410956533300"},
    "1120000000098006": {"169729410956566841", "169729410956566869", "169729410956566884", "169729410956566890", "169729410956566897"},
    "1120000000144896": {"169729410956533072"},
    "1120000000093816": {"169729410956681678"},
    "1120000000123826": {"169729410956541607"},
    "1120000000053309": {"169729410956566176"},
    "1120000000082072": {"169729410956579475"},
    "1120000000135412": {"170010885933239740"},
    "1120000000152304": {"169729410956526385"},
    "1120000000091495": {"169729410956561659", "169729410956561700"}
}


def get_email(uid):
    query = "?method=userinfo&uid=%s&userip=localhost&format=json&emails=getall" % uid
    url = BB_URL + query

    log.info("Get email from bbox: url=%s", url)

    resp = requests.get(url)
    assert resp.status_code == 200

    user = resp.json()["users"][0]
    email = user["login"] + "@" + DOMAIN

    log.info("Get email from bbox: email=%s", email)

    return email


def strip_xyhint(msg, email):
    hints = "label=SystMetkaSO:people\nlabel=symbol:delayed_message\nlabel=symbol:seen_label\nlabel=symbol:undo_message\nsave_to_sent=1\nfilters=0\nnotify=0\nemail=" + email + "\n"
    ret = "X-Yandex-Hint: " + b64encode(hints) + "\r\n"
    in_hint = False
    for line in msg.splitlines():
        if (in_hint and line.startswith((" ", "\t"))) or HINT_SPAM_RE.match(line):
            in_hint = True
            continue
        else:
            match = TM_RE.search(line)
            if match:
                ret += "X-Yandex-Hint: " + b64encode("received_date=" + match.group("ts") + "\n") + "\r\n"
            in_hint = False

        ret += line
        ret += "\r\n"
    return ret


def send_message(email, mailfrom, msg, host, port):
    log.info("Send msg: host=%s, port=%s, sender=%s, email=%s, msg_size=%s",
             host, port, mailfrom, email, len(msg))
    smtp = smtplib.SMTP(host, port, timeout=30.0)
    #  smtp.set_debuglevel(True)
    res = smtp.sendmail(mailfrom, email, msg)
    log.info("Send msg: %s", res)
    smtp.quit()


def get_message(stid):
    query = "gate/get/%s" % stid
    url = urljoin(MDS_URL, query)

    log.info("Get msg from mds: url=%s", url)

    resp = requests.get(url)
    if resp.status_code != 200:
        log.error("Failed to get email from mds, stid=%s", stid)
        raise RuntimeError("Failed to get stid")

    log.info("Get msg from mds: size=%d", len(resp.content))

    return resp.content


def setup_logging():
    handler = logging.FileHandler(LOG_FILE)
    handler.setFormatter(logging.Formatter("[%(asctime)s] %(message)s"))
    log.addHandler(handler)
    log.setLevel(logging.DEBUG)


def get_data():
    for line in sys.stdin.readlines():
        data = line.split("\t")
        uid = data[0]
        mid = data[2]
        stid = data[3]
        mfrom = data[4]
        if uid in IGNORED_UID_MID and mid in IGNORED_UID_MID[uid]:
            continue
        yield (uid, mid, stid, mfrom)


def main(host, port):
    setup_logging()
    log.info("Start")

    for uid, mid, stid, mfrom in get_data():
        try:
            log.info("Start processing stid=%s, uid=%s, mid=%s, from=%s", stid, uid, mid, mfrom)
            #  rcpt = get_email(uid)
            msg = strip_xyhint(get_message(stid), mfrom)
            send_message(mfrom, mfrom, msg, host, port)
        except Exception as e:
            log.error("Got error: %s", e)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--host", type=str, default=HOST, required=False, help="SMTP host")
    parser.add_argument("--port", type=int, default=PORT, required=False, help="SMTP port")
    args = parser.parse_args()
    main(args.host, args.port)
