#!/usr/bin/env python

import argparse
import requests
import smtplib
import logging

from urlparse import urljoin

from base64 import b64encode

LOG_FILE = "./restore_after_filters.log"
logging.getLogger().setLevel(logging.DEBUG)
log = logging.getLogger(__name__)


MDS_URL = "http://storage.mail.yandex.net:10010"
BB_URL = "http://blackbox-mail.yandex.net/blackbox"
SENDER = "maildlv-cores@yandex-team.ru"
HOST = "mxback1o.mail.yandex.net"
DOMAIN = "yandex.ru"
PORT = 25


def setup_globals(is_corp):
    global BB_URL
    global HOST
    global DOMAIN

    if is_corp:
        BB_URL = "http://blackbox.yandex-team.ru/blackbox"
        HOST = "mxbackcorp1o.mail.yandex.net"
        DOMAIN = "yandex-team.ru"


def build_xyhint(**kwargs):
    value = b64encode("\n".join("%s=%s" % (k, v) for k, v in kwargs.iteritems()))
    return "X-Yandex-Hint: " + value + "\r  \n"


def prepare_message(msg, received_date):
    hint = build_xyhint(received_date=received_date, skip_loop_prevention="1")
    log.info("Add hint: %s", hint)
    return hint + msg


def send_message(email, msg):
    log.info("Send msg: host=%s, port=%s, sender=%s, email=%s, msg_size=%s",
             HOST, PORT, SENDER, email, len(msg))
    smtp = smtplib.SMTP(HOST, PORT, timeout=30.0)
    smtp.set_debuglevel(True)
    res = smtp.sendmail(SENDER, email, msg)
    log.info("Send msg: %s", res)
    smtp.quit()


def get_email(uid):
    query = "?method=userinfo&uid=%s&userip=localhost&format=json&emails=getall" % uid
    url = BB_URL + query

    log.info("Get email from bbox: url=%s", url)

    resp = requests.get(url)
    assert resp.status_code == 200

    user = resp.json()["users"][0]
    default = next((a["address"] for a in user["address-list"] if a["default"] is True), None)
    if default:
        email = default
    else:
        email = user["login"] + "@" + DOMAIN

    log.info("Get email from bbox: email=%s", email)

    return email


def get_message(stid):
    query = "gate/get/%s" % stid
    url = urljoin(MDS_URL, query)

    log.info("Get msg from mds: url=%s", url)

    resp = requests.get(url)
    assert resp.status_code == 200

    log.info("Get msg from mds: size=%d", len(resp.content))

    return resp.content


def setup_logging():
    handler = logging.FileHandler(LOG_FILE)
    handler.setFormatter(logging.Formatter("[%(asctime)s] %(message)s"))
    log.addHandler(handler)
    log.setLevel(logging.DEBUG)


def main(is_corp, uid, stid, received_date):
    setup_globals(is_corp)
    setup_logging()
    log.info("Start with args: corp=%s, uid=%s, stid=%s, rd=%s",  is_corp, uid, stid, received_date)

    try:
        email = get_email(uid)
        msg = get_message(stid)
        msg = prepare_message(msg, received_date)
        send_message(email, msg)
    except Exception as e:
        log.error("Got error: %s", e)
        raise


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--is_corp", default=False, action="store_true", help="e.g. set or no set")
    parser.add_argument("--stid", type=str, required=True,
                        help="e.g. 320.mail:632370.E1802523:278553729310458462878420474408")
    parser.add_argument("--uid", type=int, required=True, help="e.g. 632370")
    parser.add_argument("--received_date", type=int, required=True,  help="e.g. 1551419542")
    args = parser.parse_args()

    assert args.received_date > 1500000000

    if args.is_corp:
        assert int(args.uid) > 1000000000000000
    else:
        assert int(args.uid) < 1000000000000000

    main(**dict(args._get_kwargs()))
