#!/usr/bin/env python

import argparse
import requests
import smtplib
import logging
import sys

from urlparse import urljoin

LOG_FILE = "./resend_forwards.log"
logging.getLogger().setLevel(logging.DEBUG)
log = logging.getLogger(__name__)


MDS_URL = "http://storage.mail.yandex.net:10010"
HOST = "localhost"
PORT = 5252

REMOVE_HEADERS = ("X-Yandex-Uid-Status:", "BCC:")


def send_message(rcpt, mailfrom, msg, host, port, session_id, debug=False):
    log.info("Send msg: host=%s, port=%s, sender=%s, email=%s, msg_size=%s, session_id=%s",
             host, port, mailfrom, rcpt, len(msg), session_id)
    smtp = smtplib.SMTP(host, port, timeout=30.0)
    smtp.set_debuglevel(debug)
    res = smtp.sendmail(mailfrom, rcpt, msg)
    log.info("Send msg: %s, rcpt=%s, session_id=%s", res, rcpt, session_id)
    smtp.quit()


def prepare_message(msg):
    # Remove service headers which are missed in postfix ignore header list:
    #    https://paste.yandex-team.ru/935460/text

    new_parts = [p for p in msg.split("\r\n") if not p.startswith(REMOVE_HEADERS)]
    msg = "\r\n".join(new_parts)
    # assert not [h for h in REMOVE_HEADERS if h in msg]

    return msg


def get_message(stid):
    query = "gate/get/%s" % stid
    url = urljoin(MDS_URL, query)

    log.info("Get msg from mds: url=%s", url)

    resp = requests.get(url)
    if resp.status_code != 200:
        log.error("Failed to get email from mds, stid=%s", stid)
        raise RuntimeError("Failed to get stid")

    log.info("Get msg from mds: size=%d", len(resp.content))

    return resp.content


def setup_logging():
    handler = logging.FileHandler(LOG_FILE)
    handler.setFormatter(logging.Formatter("[%(asctime)s] %(message)s"))
    log.addHandler(handler)
    log.setLevel(logging.DEBUG)


def get_data():
    for line in sys.stdin.readlines():
        mailfrom, rcpt, stid, session_id = line.split(" ")
        yield mailfrom, rcpt, stid, session_id.strip()


def main(host, port, debug):
    setup_logging()
    log.info("Start")

    for mailfrom, rcpt, stid, session_id in get_data():
        try:
            log.info("Start processing mailfrom=%s, rcpt=%s, stid=%s, session_id=%s", mailfrom, rcpt, stid, session_id)
            msg = get_message(stid)
            msg = prepare_message(msg)
            send_message(rcpt, mailfrom, msg, host, port, session_id, debug)
        except Exception as e:
            log.error("Got error: %s %s %s", e, rcpt, session_id)


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--host", type=str, default=HOST, required=False, help="SMTP host")
    parser.add_argument("--port", type=int, default=PORT, required=False, help="SMTP port")
    parser.add_argument("--debug", type=bool, default=False, required=False, help="Enable debug")
    args = parser.parse_args()
    main(args.host, args.port, args.debug)
