#include "address_parser.h"

#include <boost/bind.hpp>
#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix.hpp>


namespace NNwSmtp {

/**
 * @brief Implements "mailbox" parser.
 * @note Syntax of "mailbox" is described in rfc2822.
 *      Known issues: MAILDLV-4667
 * @see https://tools.ietf.org/html/rfc2822#section-3.4
 */
template <class Iterator, class OutputIterator>
class address_grammar : public boost::spirit::qi::grammar<Iterator>
{
public:
    explicit address_grammar(OutputIterator result) : address_grammar::base_type(m_address_list), m_result(result), m_comment_depth(0)
    {
        using boost::spirit::qi::alnum;
        using boost::spirit::qi::blank;
        using boost::spirit::qi::char_;
        using boost::spirit::qi::raw;

        // folding white space and comments
        m_fws = -(*blank >> "\r\n") >> +blank;
        m_ccontent = (char_ - char_("\\()")) | m_quoted_pair | m_comment;
        m_comment = char_('(') [boost::bind(&address_grammar::handle_comment, this, 1)]
            >> *(-m_fws >> m_ccontent) >> -m_fws
            >> char_(')') [boost::bind(&address_grammar::handle_comment, this, -1)];
        m_cfws = +(m_fws | m_comment);

        // atom
        m_atext = alnum | char_("!#$%&'*+/=?^_`{|}~-");
        m_dot_atom_text = +m_atext >> *('.' >> +m_atext);

        // quoted strings
        m_quoted_pair = '\\' >> char_;
        m_quoted_content = (char_ - char_("\\\"")) | m_quoted_pair;
        m_quoted_string = '"' >> *(-m_fws >> m_quoted_content) >> -m_fws >> '"';

        // miscellaneous tokens
        m_word = +m_atext | m_quoted_string;

        // addr-spec specification
        m_addr_spec = raw[(m_dot_atom_text | m_quoted_string) >> '@'
            >> (m_dot_atom_text | ('[' >> *(-m_fws >> m_dot_atom_text) >> -m_fws >> ']'))]
            [boost::phoenix::ref(m_address_data) = boost::spirit::qi::_1];

        // address specification
        m_display_name = raw[+m_word >> *(m_word | '.' | (m_fws >> +(m_word | '.')))] [boost::phoenix::ref(m_temp_data) = boost::spirit::qi::_1];
        m_mailbox = ((
            -(m_display_name >> -m_fws) [boost::phoenix::ref(m_name_data) = boost::phoenix::cref(m_temp_data)]
            >> (-m_cfws >> '<' >> m_addr_spec >> '>' >> -m_cfws))
            | (-m_cfws >> m_addr_spec >> -m_cfws) | m_addr_spec)
            [boost::bind(&address_grammar::handle_mailbox, this)];
        m_mailbox_list = m_mailbox >> *(-m_fws >> ',' >> -m_fws >> m_mailbox);
        m_address = (m_display_name >> -m_fws >> ':' >> -m_fws >> -(m_mailbox_list | m_cfws) >> ';' >> -m_cfws) | m_mailbox;
        m_address_list = -m_fws >> m_address >> *(-m_fws >> ',' >> -m_fws >> m_address);
    }

private:
    void handle_mailbox()
    {
        *m_result++ = std::make_pair(m_name_data, m_address_data);
        m_name_data = boost::iterator_range<Iterator>();
    }

    void handle_comment(int x)
    {
        if ((m_comment_depth += x) > 10)
            throw std::runtime_error("Comment depth exceeded");
    }

    typedef boost::spirit::qi::rule<Iterator> rule;
    rule m_fws, m_ccontent, m_comment, m_cfws;
    rule m_atext, m_dot_atom_text, m_word;
    rule m_quoted_pair, m_quoted_content, m_quoted_string;
    rule m_addr_spec, m_display_name, m_mailbox;
    rule m_mailbox_list, m_address, m_address_list;

    boost::iterator_range<Iterator> m_address_data, m_name_data, m_temp_data;

    OutputIterator m_result;
    unsigned int m_comment_depth;
};

bool parse_address(const TBufferRange& buffer, mailbox_list& result) {
    try
    {
        typedef TBufferIterator input_iterator;
        typedef std::back_insert_iterator<mailbox_list> output_iterator;
        input_iterator b = buffer.begin(), e = buffer.end();
        address_grammar<input_iterator, output_iterator> g(std::back_inserter(result));
        bool r = boost::spirit::qi::parse(b, e, g);
        return r && b == e;
    }
    catch (const std::exception &e)
    {
        return false;
    }
}

}  // namespace NNwSmtp
