#include "aliases.h"
#include "file_converter.h"
#include "log.h"
#include "utils.h"

#include <util/generic/is_in.h>

#include <algorithm>
#include <iterator>

namespace NNwSmtp {

TAliases gAliases;

void TAliases::AddAlias(const std::string& key, std::list<std::string> emails, TAliasesMap& map) const {
    auto emailsPtr = std::make_shared<TEmails>();
    emailsPtr->reserve(emails.size());
    std::move(emails.begin(), emails.end(), std::back_inserter(*emailsPtr));
    map.emplace(key, emailsPtr);

    std::string login, domain;
    if (!parse_email(key, login, domain)) {
        return;
    }
    if (IsIn(YaHosts, domain)) {
        for (const auto& host : YaHosts) {
            map.emplace(login + "@" + host, emailsPtr);
        }
    }
}

void TAliases::Init(const yplatform::ptree& conf) {
    auto yahostsRange = conf.equal_range("yahosts");
    for (auto it = yahostsRange.first; it != yahostsRange.second; ++it) {
        YaHosts.insert(it->second.data());
    }
    auto aliasMaps = conf.equal_range("maps");
    for (auto it = aliasMaps.first; it != aliasMaps.second; ++it) {
        Load(it->second.data(), Aliases);
        NWLOG_MSG(notice, "Load aliases file: name='" + it->second.data() + "'");
    }
}

void TAliases::Reload(const yplatform::ptree& conf) {
    YaHosts.clear();
    auto yahostsRange = conf.equal_range("yahosts");
    for (auto it = yahostsRange.first; it != yahostsRange.second; ++it) {
        YaHosts.insert(it->second.data());
    }
    TAliasesMap emailsMap;
    auto aliasMaps = conf.equal_range("maps");
    for (auto it = aliasMaps.first; it != aliasMaps.second; ++it) {
        Load(it->second.data(), emailsMap);
        NWLOG_MSG(notice, "Reload aliases file: name='" + it->second.data() + "'");
    }
    TWriteLock lock(Mutex);
    Aliases.swap(emailsMap);
}

void TAliases::Load(const std::string& filename, TAliasesMap& emailsMap) const {
    std::vector<std::string> lines;
    TFileConverter::ToVector(filename, lines, NConverters::TToLower());

    for (const auto& line : lines) {
        std::list<std::string> emails;
        boost::split(emails, line, boost::is_any_of(" \t,"), boost::token_compress_on);
        if (emails.size() <= 1) {
            continue;
        }
        auto keyEmail = emails.front();
        emails.pop_front();

        AddAlias(keyEmail, std::move(emails), emailsMap);
    }
}

bool TAliases::Contains(const std::string& email) const {
    TReadLock lock(Mutex);
    auto key = boost::to_lower_copy(email);
    return IsIn(Aliases, key);
}

TAliases::TEmailsPtr TAliases::Get(const std::string& email) const {
    TReadLock lock(Mutex);
    auto key = boost::to_lower_copy(email);
    auto it = Aliases.find(key);
    return (it != Aliases.end()) ? it->second : TEmailsPtr();
}

void GlobalLoadAliases(const yplatform::ptree& pt) {
    try {
        gAliases.Reload(pt.get_child("aliases"));
    } catch (const std::exception& e) {
        std::ostringstream errStream;
        errStream << "Can't reload aliases file: '" << e.what() << "'";
        throw std::logic_error(errStream.str());
    }
}

}   // namespace NNwSmtp
