#include "impl.h"
#include "utils.h"

namespace NNwSmtp::NAuth {

TAuthResponse TAuthCredentialsExtractorImpl::operator()(const AuthMethods::Login&, const std::string& input) {
    TAuthResponse resp;
    if (Data.Login.empty()) {
        extract_base64_str(input, Data.Login);
        resp.Status = Data.Login.empty() ? EStatus::BadFormed: EStatus::NeedMore;
        resp.ReplyToUser = Data.Login.empty() ? ENTER_USERNAME_BASE64 : ENTER_PASSWORD_BASE64;
        return resp;
    }

    extract_base64_str(input, Data.Password);
    resp.Status = Data.Password.empty() ? EStatus::BadFormed : EStatus::Done;
    resp.ReplyToUser = Data.Password.empty() ? ENTER_PASSWORD_BASE64 : "";
    return resp;
}

TAuthResponse TAuthCredentialsExtractorImpl::operator()(const AuthMethods::Plain&, const std::string& input) {
    std::string login;
    std::string password;
    TAuthResponse resp;
    if (extract_user_password(input, login, password)) {
        if (login.empty() || password.empty()) {
            resp.Status = EStatus::BadFormed;
            resp.ReplyToUser = ENTER_USERNAME_BASE64;
            return resp;
        }
        Data.Login = std::move(login);
        Data.Password = std::move(password);
        resp.Status = EStatus::Done;
        return resp;
    } else {
        resp.Status = EStatus::BadFormed;
        resp.ReplyToUser = ENTER_USERNAME_BASE64;
    }
    return resp;

}

TAuthResponse TAuthCredentialsExtractorImpl::operator()(const AuthMethods::XOAuth2&, const std::string& input) {
    TAuthResponse resp;

    std::string login;
    std::string token;
    if (extract_user_token(input, login, token)) {
        bool ok = !login.empty() && !token.empty();
        if (ok) {
            Data.Login = std::move(login);
            Data.Token = std::move(token);
        }
        resp.Status = ok ? EStatus::Done : EStatus::BadFormed;
        return resp;
    }
    resp.Status = EStatus::BadFormed;
    return resp;

}

TAuthResponse TAuthCredentialsExtractorImpl::operator()(const AuthMethods::NotSupported&, const std::string&) {
    TAuthResponse resp;
    resp.Status = EStatus::BadFormed;
    return resp;
}

void TAuthCredentialsExtractorImpl::SetMethod(AuthMethod method) {
    Data.AuthMethod = method;
}

std::string TAuthCredentialsExtractorImpl::GetMethod() {
    std::string method;
    boost::apply_visitor(
        TVisitors{
            [&method](const AuthMethods::Login&) { method = "LOGIN"; },
            [&method](const AuthMethods::Plain&) { method = "PLAIN"; },
            [&method](const AuthMethods::XOAuth2&) { method = "XOAUTH2"; },
            [&method](const auto&) { method = "INVALID"; },
        },
        Data.AuthMethod
    );
    return method;
}

TAuthCredentials TAuthCredentialsExtractorImpl::GetData() {
    return Data;
}

void TAuthCredentialsExtractorImpl::Reset() {
    Data = TAuthCredentials{};
}

TAuthResponse TAuthCredentialsExtractorImpl::Consume(const std::string& input) {
    return boost::apply_visitor(*this, Data.AuthMethod, boost::variant<std::string>(input));
}

}   // namespace NNwSmtp::NAuth
