#include <mail/nwsmtp/src/bb_get_result.h>
#include <mail/nwsmtp/src/blackbox/error_code.h>
#include <mail/nwsmtp/src/options.h>

namespace NNwSmtp::NBlackBox {

TCheckResult GetMailfromResult(const TErrorCode& ec, const NBlackBox::TResponse& response) {
    using NNwSmtp::NBlackBox::EError;

    TBlackBoxResult result;
    check::chk_status statusCheck;
    std::string logMsg;

    if (ec == EError::UserNotFound) {
        statusCheck = check::CHK_REJECT;
        logMsg = "empty uid";
    } else if (ec == EError::UserBlocked) {
        statusCheck = check::CHK_TEMPFAIL;
        logMsg = "User blocked.";
    } else if (ec) {
        statusCheck = check::CHK_TEMPFAIL;
        logMsg = ec.message() + ": " + response.ErrorStr;
    } else {
        statusCheck = check::CHK_ACCEPT;
        logMsg = "Success";

        result.Suid = response.Suid;
        result.Db = response.Mdb;
        result.Karma = response.Karma;
        result.KarmaStatus = response.KarmaStatus;
        result.TimeStamp = response.BornDate;
        result.Country = response.Country;
        result.Uid = response.Uid;
        result.DefaultEmail = response.DefaultEmail;
        result.PhoneConfirmed = response.PhoneConfirmed;
        result.IsHosted = response.Hosted;
        result.Domain = response.Domain;
        result.IsMaillist = response.IsMailList;
        result.Login = response.Login;
    }
    return {std::move(result), std::move(logMsg), {}, std::move(statusCheck)};
}

TCheckResult GetRcptResult(const TErrorCode& ec, const NBlackBox::TResponse& response) {
    using NNwSmtp::NBlackBox::EError;

    TBlackBoxResult result;
    check::chk_status statusCheck;
    std::string smtpResponse;
    std::string logMsg;

    if (ec == EError::UserNotFound) {
        logMsg = "User not found";
        smtpResponse = "550 5.7.1 No such user!";
    } else if (ec == EError::UserBlocked) {
        logMsg = "User blocked";
        smtpResponse = "550 5.7.1 Policy rejection on the target address";
    } else if (ec == EError::TempBanUser) {
        logMsg = "Temporary ban new user";
        smtpResponse = gconfig->blackbox.tempUserErrorReplyText;
    } else if (ec == EError::BadKarma) {
        logMsg = "User has bad karma";
        smtpResponse = "550 5.7.1 Policy rejection on the target address";
    } else if (ec) {
        logMsg = "request failed: " + ec.message() + ": " + response.ErrorStr;
        smtpResponse = gconfig->blackbox.tempUserErrorReplyText;
    } else {
        smtpResponse = "250 Ok";
        logMsg = "Success";

        if (response.Uid.empty()) {
            logMsg = "Success (user not found)";
        }

        result.Suid = response.Suid;
        result.Country = response.Country;
        result.PhoneConfirmed = response.PhoneConfirmed;
        result.Uid = response.Uid;
        result.DefaultEmail = response.DefaultEmail;
        result.OrgId = response.OrgId;
        result.IsHosted = response.Hosted;
        result.CatchAll = response.CatchAll;
        result.IsMaillist = response.IsMailList;
        result.IsCorpList = response.IsCorpList;
        result.RegistrationDate = response.RegistrationDate;
    }

    if (auto code = atoi(smtpResponse.c_str()); code >= 500) {
        statusCheck = check_rcpt_t::CHK_REJECT;
    } else if (code >= 400) {
        statusCheck = check_rcpt_t::CHK_TEMPFAIL;
    } else {
        statusCheck = check_rcpt_t::CHK_ACCEPT;
        smtpResponse = "";
    }
    return {std::move(result), std::move(logMsg), std::move(smtpResponse), std::move(statusCheck)};
}

TCheckResult GetAuthResult(const TErrorCode& ec, const NBlackBox::TResponse& response) {
    using NNwSmtp::NBlackBox::EError;

    TBlackBoxResult result;
    check::chk_status statusCheck;
    std::string smtpResponse;
    std::string logMsg;

    result.Suid = response.Suid;
    result.Uid = response.Uid;
    result.Login = response.Login;
    result.AppPasswordEnabled = response.AppPasswordEnabled;

    if (ec == EError::EmptyAuthData) {
        logMsg = "Empty auth data";
        smtpResponse = "454 4.3.0 Try again later";
    } else if (ec == EError::UserBlocked) {
        logMsg = "Invalid user or password";
        smtpResponse = "535 5.7.8 Error: authentication failed: Invalid user or password";
    } else if (ec == EError::NoAuthSuccess) {
        logMsg = "Authorization unsuccessful: " + response.ErrorStr;
        smtpResponse = "535 5.7.8 Error: authentication failed: Invalid user or password!";
    } else if (ec == EError::NoSuid) {
        logMsg = "no suid";
        smtpResponse = "535 5.7.8 Error: authentication failed: Invalid user or password!";
    } else if (ec == EError::Mdbreg) {
        logMsg = "mdbreg";
        smtpResponse = "535 5.7.8 Error: authentication failed: Invalid user or password!";
    } else if (ec == EError::ForbiddenForAssessors) {
        logMsg = "Forbidden for assessors";
        smtpResponse = "535 5.7.8 Error: authentication failed:"
            " This type of user does not have access rights to this service";
    } else if (ec == EError::NotFoundSmtpScope) {
        logMsg = "not found smtp scope";
        smtpResponse = "535 5.7.8 Error: authentication failed: The token does not have access rights to this service!";
    } else if (ec == EError::NoPddeula) {
        logMsg = "no pddeula";
        smtpResponse = "535 5.7.8 Error: authentication failed: Please accept EULA first.";
        if (!response.Domain.empty()) {
            auto country = response.Country;
            if (country == "tr") {
                country = "com.tr";
            } else if (!gconfig->blackbox.knownCountries.count(country)) {
                country = "com";
            }
            smtpResponse += " " + str(
                boost::format(gconfig->blackbox.pddEULApattern) % country % url_encode2(response.Domain)
            );
        }
    } else if (ec == EError::BadKarma) {
        logMsg = "Bad karma: " + std::to_string(response.Karma);
        smtpResponse = "535 5.7.8 Error: authentication failed:"
            " Your message looks like spam. You need to use web for"
            " sending or prove you are not a robot using the following link ";
        const auto& country = response.Country;
        auto it = gconfig->blackbox.accessRestrictedLinksMap.find(country);
        if (it != gconfig->blackbox.accessRestrictedLinksMap.end()) {
            smtpResponse += it->second;
        } else {
            auto comIt = gconfig->blackbox.accessRestrictedLinksMap.find("com");
            if (comIt == gconfig->blackbox.accessRestrictedLinksMap.end()) {
                smtpResponse += gconfig->blackbox.defaultRestrictedComLink;
            } else {
                smtpResponse += comIt->second;
            }
        }
    } else if (ec == EError::NoAccessRights) {
        logMsg = "no access rights to this service";
        smtpResponse = "535 5.7.8 Error: authentication failed: This user does not have access rights to this service";
    } else if (ec) {
        logMsg = "request failed: " + ec.message() + ": " + response.ErrorStr;
        smtpResponse = "535 5.7.8 Error: authentication failed: " + ec.message();
    } else {
        logMsg = "Success";
        smtpResponse = "235 2.7.0 Authentication successful.";
    }

    auto code = boost::lexical_cast<int>(smtpResponse.substr(0, 3));
    if (code >= 500) {
        statusCheck = check::CHK_REJECT;
    } else if (code >= 400) {
        statusCheck = check::CHK_TEMPFAIL;
    } else {
        statusCheck = check::CHK_ACCEPT;
    }
    return {std::move(result), std::move(logMsg), std::move(smtpResponse), std::move(statusCheck)};
}

} // namespace NNwSmtp
