#include "client_impl.h"
#include "errors.h"
#include "utils.h"

namespace NNwSmtp::NBigML {

namespace ph = std::placeholders;

static const std::string BIGML{"BIGML"};

namespace {

inline void LogError(const TErrorCode& ec, const TRequest& request, const TContextPtr& context,
        const std::string& msg) {
    NWLOG_CTX(error, context, BIGML, "fromEmail='" + request.FromEmail + "', toEmail='" + request.ToEmail +
        ", stat=error (" + ec.message() + (msg.empty() ? "" : (": " + msg)) + ")");
}

inline void LogSuccess(const TRequest& request, const TContextPtr& context) {
    NWLOG_CTX(notice, context, BIGML, "fromEmail='" + request.FromEmail + "', toEmail='" + request.ToEmail +
        ", stat=ok");
}

}

void TClient::Run(TContextPtr context, TRequest request, TCallback callback) {
    TCallback handler = [
        ex = boost::asio::get_associated_executor(callback, Io),
        callback = std::move(callback)
    ]
    (auto ec, auto response) mutable {
        boost::asio::post(ex, std::bind(std::move(callback), std::move(ec), std::move(response)));
    };

    HttpClient->async_run(
        context->CreateTaskContext(BIGML + ":"),
        MakeRequest(request, Config),
        MakeOptions(Config),
        std::bind(Handler, std::move(handler), std::move(request), std::move(context),
                ph::_1, ph::_2));
}

void TClient::Handler(TCallback callback, TRequest request, TContextPtr context,
    TErrorCode ec, yhttp::response response)
{
    if (ec) {
        LogError(ec, request, context, GetErrorMsg(response));
        callback(std::move(ec), {});
    } else  {
        try {
            if (response.status != 200) {
                auto ec = FromHttpStatus(response.status);
                LogError(ec, request, context, GetBigMlOrHttpErrorMsg(response));
                callback(std::move(ec), {});
            } else {
                LogSuccess(request, context);
                callback(std::move(ec), ParseResponse(response.body));
            }
        } catch (const std::exception& exc) {
            TErrorCode ec = EC_EXCEPTION;
            LogError(ec, request, context, exc.what());
            callback(std::move(ec), {});
        }
    }
}

}  // namespace NNwSmtp::NBigML
