#include "utils.h"

#include <boost/property_tree/json_parser.hpp>

namespace NNwSmtp::NBigML {

TResponse ParseResponse(const std::string& httpBody) {
    TResponse response;
    std::stringstream ss(httpBody);

    boost::property_tree::ptree tree;
    boost::property_tree::read_json(ss, tree);

    response.Code = tree.get<std::string>("response.code", "");
    response.Message = tree.get<std::string>("response.message", "");
    response.Status = tree.get<std::string>("status");
    if (response.Status == "ok") {
        for (const auto& [_, data]: tree.get_child("response.subscriptions")) {
            const auto& email = data.get<std::string>("email");
            if (email.empty()) {
                throw std::logic_error("Got empty email");
            }
            const auto& uid = data.get<std::uint64_t>("uid");
            response.Subscribers.emplace_back(email, std::to_string(uid));
        }
    }
    return response;
}

std::string BuildUrl(
    const std::string& path,
    const std::string& fromEmail,
    const std::string& toEmail)
{
    std::string url = path;

    auto questionPos = url.rfind('?');
    if (questionPos == std::string::npos) {
        url += "?";
    } else {
        url += "&";
    }

    url += "email_from=" + fromEmail;
    url += "&email_to=" + toEmail;
    url += "&version=2";
    return url;
}

std::string GetErrorMsg(const yhttp::response& resp) {
    return "status=" + std::to_string(resp.status) + " reason=" + resp.reason;
}

std::string GetBigMlOrHttpErrorMsg(const yhttp::response& resp) {
    try {
        const auto& mlResponse = ParseResponse(resp.body);
        return "status=" + std::to_string(resp.status) + " reason=" + mlResponse.Message;
    } catch (const std::exception& e) {
        return GetErrorMsg(resp);
    }
}

yhttp::request MakeRequest(const TRequest& request, const TConfig& config) {
    auto url = BuildUrl(config.path, request.FromEmail, request.ToEmail);
    return yhttp::request::GET(url);
}

ymod_httpclient::cluster_call::options MakeOptions(const TConfig& config) {
    ymod_httpclient::cluster_call::options opts;
    opts.timeouts.connect = config.connectTimeout;
    opts.timeouts.total = config.timeout;
    return opts;
}

} // namespace NNwSmtp::NBigML
