#include "bb_checks_impl.h"
#include "error_code.h"
#include "request.h"

namespace NNwSmtp::NBlackBox {

namespace ph = std::placeholders;

bool CheckOauthScope(TConfigPtr config, const TResponse& response) {
    std::set<std::string> intersection;
    std::set_intersection(
        response.Scopes.begin(),
        response.Scopes.end(),
        config->Blackbox.oauthScopes.begin(),
        config->Blackbox.oauthScopes.end(),
        std::inserter(intersection, intersection.begin())
    );
    return !intersection.empty();
}

TBBChecksImpl::TBBChecksImpl(
    TConfigPtr config,
    NClient::TBBClientPtr client
)
    : Config(std::move(config))
    , Client(std::move(client))
{}

void TBBChecksImpl::CheckRecipient(
    TContextPtr context,
    std::string email,
    std::string ip,
    bool allowList,
    bool getOrgId,
    TCallback callback
) {
    Client->UserInfo(
        std::move(context),
        BuildUserInfoRequest(
            email,
            ip,
            allowList,
            getOrgId
        ).BuildUserInfoRequest(),
        std::bind(CheckRecipientHandler, Config, std::move(callback), ph::_1, ph::_2)
    );
}

void TBBChecksImpl::CheckMailFrom(
    TContextPtr context,
    std::string email,
    std::string ip,
    TCallback callback
) {
    if (email.empty()) {
        return callback(EError::EmptySender, {});
    }

    const auto dontAllowList{false};
    const auto getOrgId{true};
    Client->UserInfo(
        std::move(context),
        BuildUserInfoRequest(
            email,
            ip,
            dontAllowList,
            getOrgId
        ).BuildUserInfoRequest(),
        std::bind(CheckMailFromHandler, Config, std::move(callback), ph::_1, ph::_2)
    );
}

void TBBChecksImpl::CheckAuth(
    TContextPtr context,
    TAuthData authInfo,
    TCallback callback
) {
    if ((authInfo.Login.empty() || authInfo.Password.empty()) && authInfo.Token.empty()) {
        return callback(EError::EmptyAuthData, {});
    }

    const bool useOauth = !authInfo.Token.empty();
    if (useOauth) {
        Client->Oauth(
            std::move(context),
            BuildAuthRequest(Config, std::move(authInfo)).BuildOauthRequest(),
            std::bind(CheckAuthHandler, Config, useOauth, std::move(callback), ph::_1, ph::_2)
        );
    } else {
        Client->Login(
            std::move(context),
            BuildAuthRequest(Config, std::move(authInfo)).BuildLoginRequest(),
            std::bind(CheckAuthHandler, Config, useOauth, std::move(callback), ph::_1, ph::_2)
        );
    }
}

void TBBChecksImpl::CheckRecipientHandler(
    TConfigPtr config,
    TCallback callback,
    TErrorCode ec,
    TResponse response
) {
    if (ec) {
        return callback(EError::HttpError, {});
    }

    if (config->CorpListUse && response.IsCorpList) {
        return callback({}, std::move(response));
    }

    if (response.Uid.empty()) {
        if (config->Blackbox.allowUnknownRcpt) {
            return callback({}, std::move(response));
        }
        return callback(EError::UserNotFound, {});
    }

    if (response.IsBlocked) {
        return callback(EError::UserBlocked, {});
    }

    if (response.Karma <= GOOD_KARMA_THRESHOLD || config->Blackbox.allowReceiveWithBadKarma) {
        return callback({}, std::move(response));
    }

    const auto now = NUtil::GetNow();
    if (std::chrono::system_clock::from_time_t(response.KarmaBanTime) > now) {
        return callback(EError::TempBanUser, {});
    }
    return callback(EError::BadKarma, {});
}

void TBBChecksImpl::CheckMailFromHandler(
    TConfigPtr config,
    TCallback callback,
    TErrorCode ec,
    TResponse response
) {
    if (ec) {
        return callback(EError::HttpError, {});
    }

    if (response.Uid.empty()) {
        if (config->Blackbox.allowUnknownSender) {
            return callback({}, std::move(response));
        }
        return callback(EError::UserNotFound, {});
    }

    if (response.IsBlocked) {
        return callback(EError::UserBlocked, {});
    }

    callback({}, std::move(response));
}

void TBBChecksImpl::CheckAuthHandler(
    TConfigPtr config,
    bool oauth,
    TCallback callback,
    TErrorCode ec,
    TResponse response
) {
    if (ec) {
        return callback(EError::HttpError, {});
    }

    if (response.IsBlocked) {
        return callback(EError::UserBlocked, {});
    }

    if (!response.AuthSuccess) {
        return callback(EError::NoAuthSuccess, std::move(response));
    }

    if (config->Blackbox.denyAuthForAssessors && response.IsAssessor) {
        return callback(EError::ForbiddenForAssessors, {});
    }

    if (oauth && !CheckOauthScope(config, response)) {
        response.AuthSuccess = false;
        return callback(EError::NotFoundSmtpScope, {});
    }

    if (response.Hosted && !response.HasPDDsid) {
        response.AuthSuccess = false;
        return callback(EError::NoPddeula, std::move(response));
    }

    if (response.Suid == 0) {
        response.AuthSuccess = false;
        return callback(EError::NoSuid, {});
    }

    if (response.Mdb == "mdbreg") {
        response.AuthSuccess = false;
        return callback(EError::Mdbreg, {});
    }

    if (!response.PhoneConfirmed &&
        (response.Karma == GOOD_KARMA_THRESHOLD || response.Karma == BAD_KARMA) &&
        !response.HasPDDsid)
    {
        response.AuthSuccess = false;
        return callback(EError::BadKarma, std::move(response));
    }

    callback({}, std::move(response));
}

} // namespace NNwSmtp::NBlackBox
