#include "bb_client_impl.h"
#include "client_types_reflection.h"

#include <mail/nwsmtp/src/log.h>
#include <mail/yreflection/include/yamail/data/deserialization/json_reader.h>

#include <ymod_httpclient/call.h>

#include <format>

namespace NNwSmtp::NBlackBox::NClient {

namespace {

void LogSuccess(const TContextPtr& context) {
    NWLOG_CTX(notice, context, "", "bbatt stat=ok");
}

void LogError(const TContextPtr& context, const TErrorCode& errc, const std::string& reason) {
    const auto format = reason.empty() ? "bbatt stat=error ({})" : "bbatt stat=error ({}: {})";
    NWLOG_CTX(error, context, "", std::format(format, errc.message(), reason));
}

template <typename TActualResponse>
void HandleHttpResponse(
    TContextPtr context,
    TClientImpl::TResponseCallback callback,
    TErrorCode errc,
    yhttp::response httpResponse
) {
    if (errc) {
        LogError(context, errc, httpResponse.reason);
        return callback(std::move(errc), {});
    }

    if (httpResponse.status != 200) {
        errc = ymod_httpclient::http_error::server_status_error;
        LogError(context, errc, httpResponse.reason.empty() ? std::to_string(httpResponse.status) : httpResponse.reason);
        return callback(std::move(errc), {});
    }

    using TResponseData = typename TActualResponse::TResponseData;
    TResponseData bbResponse;
    try {
        yamail::data::deserialization::fromJson<TResponseData>(httpResponse.body, bbResponse);
    } catch (const std::exception& e) {
        errc = yhttp::errc::parse_response_error;
        LogError(context, errc, e.what());
        return callback(std::move(errc), {});
    }

    if (bbResponse.Exception.has_value()) {
        errc = ymod_httpclient::http_error::server_response_error;
        std::string reason = bbResponse.Error.value_or(httpResponse.reason);
        if (reason.empty()) {
            reason = std::format("BlackBox exception value: {}", bbResponse.Exception->Value);
        }
        LogError(context, errc, std::move(reason));
        return callback(std::move(errc), {});
    }

    try {
        TActualResponse response{bbResponse};
        LogSuccess(context);
        return callback(std::move(errc), std::move(response.Result));
    } catch (const std::exception& e) {
        errc = yhttp::errc::parse_response_error;
        LogError(context, errc, e.what());
        return callback(std::move(errc), {});
    }
}

std::pair<TErrorCode, std::string> GetTvmServiceTicket(const TContextPtr& context, ymod_tvm::tvm2_module& tvm2Module) {
    static const std::string tvmServiceName = "blackbox";
    std::string ticket;
    auto errorCode = tvm2Module.get_service_ticket(tvmServiceName, ticket);
    if (errorCode) {
        NWLOG_CTX(error, context, "", std::format("Failed to get TVM2 service ticket ({})", errorCode.message()));
    }
    return std::make_pair(std::move(errorCode), std::move(ticket));
}

template <typename TActualResponse>
void AsyncRun(
    TContextPtr context,
    THttpRequest request,
    TClientImpl::THttpClient httpClient,
    ymod_tvm::tvm2_module& tvm2Module,
    boost::asio::io_context& io,
    TClientImpl::TResponseCallback callback
) {
    auto [tvmErrorCode, tvmServiceTicket] = GetTvmServiceTicket(context, tvm2Module);
    if (tvmErrorCode) {
        return callback(std::move(tvmErrorCode), {});
    }

    auto clientContext = context->CreateTaskContext("BB:");
    auto handler = [ex = io.get_executor(), context = std::move(context), callback = std::move(callback)]
        (TErrorCode errc, yhttp::response response) {
            boost::asio::post(
                ex,
                std::bind(
                    HandleHttpResponse<TActualResponse>,
                    context,
                    callback,
                    errc,
                    response
                )
            );
        };
    httpClient->async_run(std::move(clientContext), MakeRequest(std::move(request), std::move(tvmServiceTicket)), std::move(handler));
}

ymod_httpclient::headers_param MakeHeaders(std::string tvmServiceTicket, bool addContentTypeHeader) {
    if (addContentTypeHeader) {
        return ymod_httpclient::headers_param {
            {"Content-Type", "application/x-www-form-urlencoded"},
            {"X-Ya-Service-Ticket", std::move(tvmServiceTicket)},
        };
    }
    return ymod_httpclient::headers_param {
        {"X-Ya-Service-Ticket", std::move(tvmServiceTicket)},
    };
}

} // namespace anonymous

yhttp::request MakeRequest(THttpRequest request, std::string tvmServiceTicket) {
    request.Url.insert(0, "/");

    if (request.Body.empty()) {
        return yhttp::request::GET(
            std::move(request.Url),
            MakeHeaders(std::move(tvmServiceTicket), false)
        );
    }

    return yhttp::request::POST(
        std::move(request.Url),
        MakeHeaders(std::move(tvmServiceTicket), true),
        std::move(request.Body)
    );
}

TClientImpl::TClientImpl(
    THttpClient httpClient,
    TTvm2Module tvm2Module,
    boost::asio::io_context& io
)
    : HttpClient(std::move(httpClient))
    , Tvm2Module(std::move(tvm2Module))
    , Io(io)
{}

void TClientImpl::UserInfo(TContextPtr context, THttpRequest request, TResponseCallback callback) {
    AsyncRun<TInfoResponse>(
        std::move(context),
        std::move(request),
        HttpClient,
        *Tvm2Module,
        Io,
        std::move(callback)
    );
}

void TClientImpl::Login(TContextPtr context, THttpRequest request, TResponseCallback callback) {
    AsyncRun<TLoginResponse>(
        std::move(context),
        std::move(request),
        HttpClient,
        *Tvm2Module,
        Io,
        std::move(callback)
    );
}

void TClientImpl::Oauth(TContextPtr context, THttpRequest request, TResponseCallback callback) {
    AsyncRun<TOAuthResponse>(
        std::move(context),
        std::move(request),
        HttpClient,
        *Tvm2Module,
        Io,
        std::move(callback)
    );
}

} // namespace NNwSmtp::NBlackBox::NClient
