#pragma once

#include <boost/system/error_code.hpp>

namespace NNwSmtp::NBlackBox {

enum class EError {
    Ok,
    BbTempUserError,
    UserNotFound,
    UserBlocked,
    TempBanUser,
    BadKarma,
    EmptySender,
    EmptyAuthData,
    ForbiddenForAssessors,
    Mdbreg,
    NoSuid,
    NoPddeula,
    NotFoundSmtpScope,
    NoAccessRights,
    NoAuthSuccess,
    HttpError,
};

} // namespace NNwSmtp::NBlackBox

namespace boost::system {

template <>
struct is_error_code_enum<NNwSmtp::NBlackBox::EError> : std::true_type {};

} // namespace boost::system

namespace NNwSmtp::NBlackBox {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "nwsmtp_blackbox";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::BbTempUserError:
                return "Requested action to blackbox aborted";
            case EError::UserNotFound:
                return "User not found";
            case EError::UserBlocked:
                return "User blocked";
            case EError::TempBanUser:
                return "Temporary ban user";
            case EError::BadKarma:
                return "User has bad karma";
            case EError::EmptySender:
                return "Empty sender";
            case EError::EmptyAuthData:
                return "Empty auth data";
            case EError::ForbiddenForAssessors:
                return "Forbidden for assessors";
            case EError::Mdbreg:
                return "Mdbreg";
            case EError::NoSuid:
                return "No suid";
            case EError::NoPddeula:
                return "No pddeula";
            case EError::NotFoundSmtpScope:
                return "Not found smtp scope";
            case EError::NoAccessRights:
                return "Not found smtp scope";
            case EError::NoAuthSuccess:
                return "No auth success";
            case EError::HttpError:
                return "Http client error";
        }
        return "Unknown error code: " + std::to_string(value);
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

} // namespace NNwSmtp::NBlackBox
