#include "request.h"

#include <boost/algorithm/string/join.hpp>
#include <butil/network/idn.h>
#include <ymod_httpclient/url_encode.h>

#include <vector>

namespace NNwSmtp::NBlackBox {

namespace {

class TFields {
public:
    TFields& operator<<(std::string field) {
        Fields.push_back(std::move(field));
        return *this;
    }

    std::string Get() const {
        return boost::algorithm::join(Fields, ",");
    }

private:
    std::vector<std::string> Fields;
};

} // namespace anonymous

NClient::THttpRequest TRequest::BuildUserInfoRequest() const {
    return NClient::THttpRequest {
        .Url = yhttp::url_encode({
            {"method",     "userinfo"},
            {"format",     Format},
            {"login",      Login},
            {"sid",        Sid},
            {"userip",     UserIp},
            {"emails",     Emails},
            {"dbfields",   DbFields},
            {"attributes", Attributes},
        }),
    };
}

NClient::THttpRequest TRequest::BuildLoginRequest() const {
    return NClient::THttpRequest {
        .Url = yhttp::url_encode({
            {"method",     "login"},
            {"format",     Format},
            {"login",      Login},
            {"sid",        Sid},
            {"authtype",   AuthType},
            {"userip",     UserIp},
            {"emails",     Emails},
            {"dbfields",   DbFields},
            {"attributes", Attributes},
        }),
        .Body = yhttp::form_encode({
            {"password", Password},
        }),
    };
}

NClient::THttpRequest TRequest::BuildOauthRequest() const {
    return NClient::THttpRequest {
        .Url = yhttp::url_encode({
            {"method",     "oauth"},
            {"format",     Format},
            {"userip",     UserIp},
            {"emails",     Emails},
            {"dbfields",   DbFields},
            {"attributes", Attributes},
        }),
        .Body = yhttp::form_encode({
            {"oauth_token", OauthToken},
        }),
    };
}

TRequest BuildUserInfoRequest(
    const std::string& login,
    const std::string& userIp,
    bool allowList,
    bool getOrgId
) {
    TFields dbFields;
    dbFields << NClient::NDbFields::RegDate
             << NClient::NDbFields::Suid
             << NClient::NDbFields::LoginRule2
             << NClient::NDbFields::BornDate
             << NClient::NDbFields::Login
             << NClient::NDbFields::Country
             << NClient::NDbFields::PhoneConfirmedUid;
    if (allowList) {
        dbFields << NClient::NDbFields::Suid1000;
    }

    TFields attributes;
    attributes << NClient::NAttributes::IsMailList;
    if (getOrgId) {
        attributes << NClient::NAttributes::OrgId;
    }

    return TRequest {{
        .Login = login,
        .UserIp = userIp,
        .DbFields = dbFields.Get(),
        .Attributes = attributes.Get(),
    }};
}

TRequest BuildAuthRequest(TConfigPtr config, TAuthData authData) {
    TFields dbFields;
    dbFields << NClient::NDbFields::Suid102
             << NClient::NDbFields::LoginRule8
             << NClient::NDbFields::LoginRule2
             << NClient::NDbFields::Mdb
             << NClient::NDbFields::Country
             << NClient::NDbFields::PhoneConfirmedUid;

    TFields attributes;
    if (config->Blackbox.denyAuthForAssessors) {
        attributes << NClient::NAttributes::IsAssessor;
    }

    if (authData.Token.empty()) {
        dbFields << NClient::NDbFields::Suid << NClient::NDbFields::Login;
        if (config->AuthSettingsOptsUse) {
            attributes << NClient::NAttributes::AppPasswordEnabled;
        }
        return TRequest {{
            .Login = std::move(removePlusFromEmail(authData.Login)),
            .UserIp = std::move(authData.Ip),
            .Password = std::move(authData.Password),
            .DbFields = dbFields.Get(),
            .Attributes = attributes.Get(),
        }};
    } else {
        dbFields << NClient::NDbFields::Suid2 << NClient::NDbFields::Login2;
        return TRequest {{
            .UserIp = std::move(authData.Ip),
            .OauthToken = std::move(authData.Token),
            .DbFields = dbFields.Get(),
            .Attributes = attributes.Get(),
        }};
    }
}

} // namespace NNwSmtp::NBlackBox
