#include "response.h"
#include "utils.h"

#include <mail/nwsmtp/src/utils.h>

#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/property_tree/json_parser.hpp>
#include <butil/network/idn.h>

#include <chrono>
#include <ctime>
#include <stdexcept>

namespace NNwSmtp::NBlackBox {

namespace {

std::time_t DateTimeToUtcTimestamp(const std::string& dateTime, std::time_t def) {
    int y = 0, mo = 0, d = 0, h = 0, mi = 0, s = 0;
    if (std::sscanf(dateTime.c_str(), "%d-%d-%d %d:%d:%d", &y, &mo, &d, &h, &mi, &s) == EOF) {
        return def;
    }
    auto calendarTime = std::tm {
        .tm_sec = s,
        .tm_min = mi,
        .tm_hour = h,
        .tm_mday = d,
        .tm_mon = mo - 1,
        .tm_year = y - 1900,
        .tm_wday = 0,
        .tm_yday = 0,
        .tm_isdst = -1,
    };
    return std::mktime(&calendarTime);
}

} // namespace anonymous

TBaseResponse::TBaseResponse(const NClient::TCommonResponse& responseData)
    : Result{}
    , ErrorMsg{responseData.Error.value_or("")}
{
    if (responseData.Exception.has_value()) {
        throw std::runtime_error(ErrorMsg);
    }
}

const std::string& TBaseResponse::GetErrorMsg() const {
    return ErrorMsg;
}

TInfoResponse::TInfoResponse(const NClient::TUserInfoResponseData& responseData)
    : TBaseResponse{responseData}
{
    if (!responseData.Users.has_value() || responseData.Users->empty()) {
        throw std::runtime_error("Empty users array in userinfo response");
    }
    ParseUserInfo(responseData.Users->front());
}

TLoginResponse::TLoginResponse(const NClient::TLoginResponseData& responseData)
    : TBaseResponse{responseData}
{
    EStatus status;
    if (responseData.Status.has_value()) {
        status = static_cast<EStatus>(responseData.Status->Id);
    } else if (responseData.LoginStatus.has_value() && responseData.PasswordStatus.has_value()) {
        const auto accountStatus = static_cast<EAccountStatus>(responseData.LoginStatus->Id);
        const auto passwordStatus = static_cast<EPasswordStatus>(responseData.PasswordStatus->Id);
        if (accountStatus == EAccountStatus::AccValid && passwordStatus == EPasswordStatus::PwdValid) {
            status = EStatus::Valid;
        } else if (accountStatus == EAccountStatus::AccDisabled) {
            status = EStatus::Disabled;
        } else if (accountStatus == EAccountStatus::AccAlienDomain) {
            status = EStatus::AlienDomain;
        } else if (passwordStatus == EPasswordStatus::PwdCompromised) {
            status = EStatus::Compromised;
        } else if (responseData.BruteforcePolicy.has_value() && responseData.BruteforcePolicy->Value == "captcha") {
            status = EStatus::ShowCaptcha;
        } else if (responseData.BruteforcePolicy.has_value() && responseData.BruteforcePolicy->Value == "password_expired") {
            status = EStatus::Expired;
        } else {
            status = EStatus::Invalid;
        }
    } else {
        throw std::runtime_error("Bad blackbox response: missing login or password status");
    }

    if (responseData.Comment.has_value()) {
        ErrorMsg = responseData.Comment.value();
    }

    ParseUserInfo(responseData);

    if (status == TLoginResponse::EStatus::Valid) {
        Result.AuthSuccess = true;
    } else {
        Result.AuthSuccess = false;
        Result.ErrorStr = GetErrorMsg();
    }
}

TOAuthResponse::TOAuthResponse(const NClient::TOauthResponseData& responseData)
    : TBaseResponse{responseData}
{
    EStatus status;
    if (!responseData.Status.has_value()) {
        throw std::runtime_error("Bad blackbox response: missing status.id");
    }
    status = static_cast<EStatus>(responseData.Status->Id);

    std::optional<std::string> scope;
    if (responseData.Oauth.has_value()) {
        scope = responseData.Oauth->Scope;
    }

    ParseUserInfo(responseData);

    if (status == TOAuthResponse::EStatus::Valid) {
        Result.AuthSuccess = true;
        if (scope.has_value()) {
            boost::algorithm::split(Result.Scopes, scope.value(), boost::is_any_of(" "));
        }
    } else {
        Result.AuthSuccess = false;
        Result.ErrorStr = GetErrorMsg();
    }
}

void TBaseResponse::ParseUserInfo(const NClient::TCommonUserInfo& userInfo) {
    if (userInfo.AddressList.has_value() && !userInfo.AddressList->empty()) {
        const auto& firstAddressInfo = userInfo.AddressList->front();
        Result.DefaultEmail = idna::decode(firstAddressInfo.Address);
    }

    if (userInfo.Attributes.has_value()) {
        const auto& attributes = userInfo.Attributes.value();
        if (auto mailListIt = attributes.find(NClient::NAttributes::IsMailList); mailListIt != attributes.end()) {
            Result.IsMailList = SafeCast(mailListIt->second, Result.IsMailList);
        }
        if (auto appPasswordEnabledIt = attributes.find(NClient::NAttributes::AppPasswordEnabled); appPasswordEnabledIt != attributes.end()) {
            Result.AppPasswordEnabled = SafeCast(appPasswordEnabledIt->second, Result.AppPasswordEnabled);
        }
        if (auto isAccessorIt = attributes.find(NClient::NAttributes::IsAssessor); isAccessorIt != attributes.end()) {
            Result.IsAssessor = SafeCast(isAccessorIt->second, Result.IsAssessor);
        }
        if (auto orgIdIt = attributes.find(NClient::NAttributes::OrgId); orgIdIt != attributes.end()) {
            Result.OrgId = orgIdIt->second;
        }
    }

    if (userInfo.DbFields.has_value()) {
        const auto& dbFields = userInfo.DbFields.value();
        if (auto suidIt = dbFields.find(NClient::NDbFields::Suid); suidIt != dbFields.end() && suidIt->second.has_value()) {
            Result.Suid = SafeCast(suidIt->second.value(), Result.Suid);
        }
        if (auto suid2It = dbFields.find(NClient::NDbFields::Suid2); suid2It != dbFields.end() && suid2It->second.has_value()) {
            Result.Suid = SafeCast(suid2It->second.value(), Result.Suid);
        }
        if (auto loginIt = dbFields.find(NClient::NDbFields::Login); loginIt != dbFields.end() && loginIt->second.has_value()) {
            Result.Login = loginIt->second.value();
        }
        if (auto login2It = dbFields.find(NClient::NDbFields::Login2); login2It != dbFields.end() && login2It->second.has_value()) {
            Result.Login = login2It->second.value();
        }
        if (auto countryIt = dbFields.find(NClient::NDbFields::Country); countryIt != dbFields.end() && countryIt->second.has_value()) {
            Result.Country = countryIt->second.value();
        }
        if (auto phoneConfirmedUidIt = dbFields.find(NClient::NDbFields::PhoneConfirmedUid); phoneConfirmedUidIt != dbFields.end() && phoneConfirmedUidIt->second.has_value()) {
            Result.PhoneConfirmed = !phoneConfirmedUidIt->second->empty();
        }
        if (auto suid102It = dbFields.find(NClient::NDbFields::Suid102); suid102It != dbFields.end() && suid102It->second.has_value()) {
            Result.HasPDDsid = !suid102It->second->empty();
        }
        if (auto suid1000It = dbFields.find(NClient::NDbFields::Suid1000); suid1000It != dbFields.end() && suid1000It->second.has_value()) {
            Result.IsCorpList = suid1000It->second->empty();
        }
        if (auto loginRule2It = dbFields.find(NClient::NDbFields::LoginRule2); loginRule2It != dbFields.end() && loginRule2It->second.has_value()) {
            Result.IsBlocked |= (loginRule2It->second.value() == "0");
        }
        if (auto loginRule8It = dbFields.find(NClient::NDbFields::LoginRule8); loginRule8It != dbFields.end() && loginRule8It->second.has_value()) {
            Result.IsBlocked |= (SafeCast<std::uintmax_t>(loginRule8It->second.value(), 0) & 0x04);
        }
        if (auto regDateIt = dbFields.find(NClient::NDbFields::RegDate); regDateIt != dbFields.end() && regDateIt->second.has_value()) {
            Result.RegistrationDate = DateTimeToUtcTimestamp(regDateIt->second.value(), Result.RegistrationDate);
        }
        if (auto bornDateIt = dbFields.find(NClient::NDbFields::BornDate); bornDateIt != dbFields.end() && bornDateIt->second.has_value()) {
            Result.BornDate = DateTimeToUtcTimestamp(bornDateIt->second.value(), Result.BornDate);
        }
        if (auto mdbIt = dbFields.find(NClient::NDbFields::Mdb); mdbIt != dbFields.end() && mdbIt->second.has_value()) {
            Result.Mdb = mdbIt->second.value();
        }
    }

    if (userInfo.Karma.has_value()) {
        const auto& karma = userInfo.Karma.value();
        Result.Karma = karma.Value;
        Result.KarmaBanTime = karma.AllowUntil.value_or(Result.KarmaBanTime);
    }

    if (userInfo.KarmaStatus.has_value()) {
        Result.KarmaStatus = userInfo.KarmaStatus->Value;
    }

    if (userInfo.Uid.has_value()) {
        const auto& uid = userInfo.Uid.value();
        Result.Hosted = uid.IsHosted.value_or(Result.Hosted);
        if (uid.Domain) {
            Result.Domain = idna::decode(uid.Domain.value());
        }
        if (uid.CatchAll.value_or(false) || !uid.DomainEna.value_or("").empty()) {
            Result.CatchAll = true;
        }
        Result.Uid = uid.Value.value_or(Result.Uid);
    }
}

} // namespace NNwSmtp::NBlackBox
