#include "control_from.h"

#include <mail/nwsmtp/src/bb_get_result.h>

#include <boost/format.hpp>

namespace NNwSmtp::NControlFrom {

namespace ph = std::placeholders;

namespace {

inline bool is_in_white_list(const TContextPtr& context, const TBlackBoxResultOpt& senderInfo) {
    bool isDomainInWhiteList = false;
    bool isUidInWhiteList = false;

    if (senderInfo->IsHosted) {
        isDomainInWhiteList = gconfig->msgOpts.ControlFromOpts.WhiteList.ContainsDomain(senderInfo->Domain);
    } else {
        isUidInWhiteList = gconfig->msgOpts.ControlFromOpts.WhiteList.ContainsUid(senderInfo->Uid);
    }
    NWLOG_CTX(notice, context, "HDR-FROM-WLIST",
        "domain_in_white_list=" + std::to_string(isDomainInWhiteList) +
        ", uid_in_white_list=" + std::to_string(isUidInWhiteList));
    return isDomainInWhiteList || isUidInWhiteList;
}

}

void control_from::start(TRequest request, TCallback callback) {
    const auto& sender = request.Envelope->header_sender_;
    const auto& from = request.Envelope->header_from_;

    TCallback handle =
        [this, callback = std::move(callback), sessionId = request.SessionId, envelope = request.Envelope]
        (auto ec, auto response) {
            if (ec == EError::Reject && is_in_white_list(context, envelope->senderInfo)) {
                ec = EError::WhiteList;
            }

            NWLOG_CTX(notice, context, "HDR-FROM", "status=" + ec.message() +
                ", header='" + std::string(header.begin(), header.end()) + "', email='" +  +
                "', error='" + response.Error + "'");

            boost::asio::post(io, std::bind(callback, ec, std::move(response)));
    };

    if (!sender && !from) {
        return handle(EError::Accept, {});
    }

    if (sender) {
        header = sender;
        mbodyEmail = parseEmail(sender);
        if (mbodyEmail.empty()) {
            return handle(
                EError::Reject,
                {"failed to parse 'Sender' header", "550 5.7.0 Sender header syntax is invalid"});
        }
    }

    if (from) {
        auto email = parseEmail(from);
        if (email.empty()) {
            header = from;
            mbodyEmail = email;
            return handle(
                EError::Reject,
                {"failed to parse 'From' header", "550 5.7.0 From header syntax is invalid"}
            );
        }
        mbodyEmail = mbodyEmail.empty() ? email : mbodyEmail;
        header = header.empty() ? from : header;
    }

    std::string local, domain;
    parse_email(mbodyEmail, local, domain);

    if (!request.Envelope->senderInfo) {
        return handle(EError::Accept, {});
    }

    bbChecks->CheckMailFrom(
        context,
        mbodyEmail,
        request.RemoteIp,
        std::bind(&control_from::handle_mailfrom_check, shared_from_this(),
            std::move(mbodyEmail), std::move(request), std::move(handle), ph::_1, ph::_2)
    );
}

std::string control_from::parseEmail(const TBufferRange& value) {
    mailbox_list mailboxes;
    if (parse_address(value, mailboxes) && !mailboxes.empty()) {
        return std::string(mailboxes.front().second.begin(), mailboxes.front().second.end());
    } else {
        return {};
    }
}

void control_from::handle_mailfrom_check(
    std::string mailfrom,
    TRequest request,
    TCallback callback,
    TErrorCode ec,
    NBlackBox::TResponse response
) {

    auto checkResult = NBlackBox::GetMailfromResult(ec, response);
    NWLOG_CTX(notice, context, "BB-MAILFROM", "mailfrom='" + mailfrom + "', status='" + checkResult.LogMsg + "'");

    if (checkResult.StatusCheck == check::CHK_REJECT) {
        return callback(EError::Reject,
            {
                "reject from blackbox",
                "550 5.7.0 Sender or From header address rejected: not owned by authorized user"
            }
        );
    }
    if (checkResult.StatusCheck == check::CHK_TEMPFAIL) {
        return callback(EError::Error, {"tempfail from blackbox", temp_error});
    }

    if (checkResult.BlackBoxResult.Uid == request.Envelope->senderInfo->Uid) {
        return callback(EError::Accept, TResponse {});
    }

    if (gconfig->bigMlOpts.use && checkResult.BlackBoxResult.IsMaillist) {
        return bigMl->Run(
            context,
            {mailfrom, mailfrom},
            std::bind(&control_from::handle_resolve_maillist,
                shared_from_this(), std::move(request), std::move(callback), ph::_1, ph::_2));
    }
    return callback(
        EError::Reject,
        {
            "not owned by the authorized user",
            "550 5.7.0 Sender or From header address rejected: not owned by authorized user"
        }
    );
}

void control_from::handle_resolve_maillist(
    TRequest request,
    TCallback callback,
    TErrorCode ec,
    NBigML::TResponse resp
) {
    bool isUidInSubscribers = false;
    if (!ec) {
        isUidInSubscribers = std::any_of(resp.Subscribers.cbegin(), resp.Subscribers.cend(),
            [&request] (const auto& subscriber) {
                return subscriber.second == request.Envelope->senderInfo->Uid;
            });
    }

    NWLOG_CTX(notice, context, "HDR-FROM-BIGML",
        "status=" + ec.message() + ", is_uid_in_subscribers=" + std::to_string(isUidInSubscribers));

    if (ec) {
        return callback(EError::Error, {"tempfail from bigml", temp_error});
    }

    if (isUidInSubscribers) {
        return callback(EError::Accept, {});
    }
    return callback(
        EError::Reject,
        {
            "not owned by the authorized user",
            "550 5.7.0 Sender or From header address rejected: not owned by authorized user"
        }
    );
}

}  // namespace NNwSmtp::NControlFrom
