#pragma once

#include "types.h"

#include <mail/library/dsn/composer.hpp>

#include <mail/nwsmtp/src/avir/client.h>
#include <mail/nwsmtp/src/control_from/types.h>
#include <mail/nwsmtp/src/dmarc.h>
#include <mail/nwsmtp/src/dkim/adkim.h>
#include <mail/nwsmtp/src/dkim/dkim_module.h>
#include <mail/nwsmtp/src/smtp_client/smtp_client.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/recognizer/recognizer.h>
#include <mail/nwsmtp/src/router/router.h>
#include <mail/nwsmtp/src/so/client.h>
#include <mail/nwsmtp/src/so/types.h>
#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/yarm/client.h>

#include <boost/asio.hpp>

#include <memory>
#include <list>

namespace NNwSmtp::NAsyncDlv {

class TAsyncDeliveryImpl : public std::enable_shared_from_this<TAsyncDeliveryImpl> {
public:
    TAsyncDeliveryImpl(
        TCallback callback,
        TContextPtr context,
        TRequest dlvRequest,
        SmtpClient::TSmtpClientPtr smtpClient,
        NSO::TSOClientPtr soClient,
        NAvir::TAvirCheckClientPtr avirCheckClient,
        TRecognizerPtr recognizer,
        NYarm::TClientPtr yarmClient,
        TRouterPtr router,
        boost::asio::io_context& io
    )   : Callback(std::move(callback))
        , Context(std::move(context))
        , DlvRequest(std::move(dlvRequest))
        , SmtpClient(std::move(smtpClient))
        , SoClient(std::move(soClient))
        , Recognizer(std::move(recognizer))
        , AvirCheckClient(std::move(avirCheckClient))
        , YarmClient(std::move(yarmClient))
        , Router(std::move(router))
        , Io(io) {}

    void Run(TErrorCode ec = {}, std::string answer = {});

private:
    void PrepareHeaders();
    void PerformDecycling();

    void PrepareDelivery(TErrorCode ec = {}, std::string answer = {});
    void ControlFrom();
    void HandleControlFrom(TErrorCode ec, NControlFrom::TResponse response);
    void DkimCheck();
    void HandleDkimCheck(dkim_check::Output output);
    void DmarcCheck();
    void HandleDmarcCheck(std::optional<dmarc::record> record);
    void DkimSign();
    void HandleDkimSign(std::string error, dkim_sign::Output output);
    void SoCheck();
    void HandleSoCheck(TErrorCode ec, NSO::TResponse response);
    void AvirCheckData();
    void HandleAvirCheck(TErrorCode ec, NAvir::TStatus status);
    void AddAuthResultsHeader();
    void PrepareAlteredMessage();
    void GetRpopAuth();
    void HandleRpopAuth(TErrorCode ec, NYarm::TResponse response);
    void Route();
    void HandleRoute(TErrorCode ec);

    void Delivery(TErrorCode ec = {}, std::string answer = {});
    void Back();
    void Local();
    void External(const SmtpPoint& point, const SmtpTimeouts& output);
    void Dsn(dsn::composer::type_t type, TErrorCode ec = {}, std::string answer = {});
    envelope_ptr MakeDsnEnvelope(dsn::composer::type_t type);
    std::list<TRcpt> GetRcpts(std::function<bool(DomainType domainType)> pred);

    void LogAnalytics();

    TCallback Callback;
    TContextPtr Context;
    TRequest DlvRequest;

    SmtpClient::TSmtpClientPtr SmtpClient;
    NSO::TSOClientPtr SoClient;
    TRecognizerPtr Recognizer;

    envelope_ptr DsnEnvelope;

    boost::asio::coroutine RunCoroutine;
    boost::asio::coroutine DsnCoroutine;
    boost::asio::coroutine DeliveryCoroutine;
    boost::asio::coroutine PrepareDeliveryCoroutine;

    std::list<TRcpt> CurrentRcpts;

    SmtpPoint SenderDependentRelay;
    SmtpPoint ForeignMxRelay;

    std::string FromDomain;
    std::string RequestId;

    NSO::EResolution SpamHeaderStatus;
    std::string AuthResultsHeaderValue;
    bool SoCaptcha = false;
    bool SkipSoCheck = false;

    NAvir::TStatus VirusHeaderStatus;
    bool SkipAvirCheck = false;
    NAvir::TAvirCheckClientPtr AvirCheckClient;

    NYarm::TClientPtr YarmClient;
    std::string AuthLogin;
    std::string AuthPassword;
    bool AuthOauth = false;

    bool SkipDkimSignIfError = AllowSkipDkimSign();
    bool HasDkimHeaders = false;
    DkimOptions::dkim_mode DkimMode = GetDkimMode();
    dkim_check::dkim_status DkimStatus = dkim_check::none;
    std::string DkimIdentity;
    std::string DkimDomain;
    std::list<std::string> DkimDomains;

    std::shared_ptr<dmarc::checker> DmarcChecker;
    dmarc::record::policy DmarcPolicy = dmarc::record::UNKNOWN;
    std::string DmarcDomain;

    std::string MailerHdrName;
    std::string Mailer;

    TRouterPtr Router;

    boost::asio::io_context& Io;
};

} // namespace NwSmtp::NAsyncDlv
